/*
 *    PlaceholderDirectoryEditor.java
 *    Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.goe;

import java.awt.FontMetrics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JComponent;

import adams.core.io.PlaceholderDirectory;
import adams.core.option.AbstractOption;
import adams.gui.chooser.BaseDirectoryChooser;

/**
 * A PropertyEditor for PlaceholderDirectory objects that lets the user select
 * a directory.
 * <p/>
 * Based on <code>weka.gui.FileEditor</code>.
 *
 * @author Len Trigg (trigg@cs.waikato.ac.nz)
 * @version $Revision: 1737 $
 * @see weka.gui.PlaceholderDirectoryEditor
 */
public class PlaceholderDirectoryEditor
  extends AbstractPropertyEditorSupport
  implements CustomStringRepresentationHandler {

  /** The directory chooser used for selecting dirs. */
  protected BaseDirectoryChooser m_DirChooser;

  /**
   * Returns the file as string.
   *
   * @param option	the current option
   * @param object	the file object to convert
   * @return		the generated string
   */
  public static String toString(AbstractOption option, Object object) {
    return ((PlaceholderDirectory) object).getPath();
  }

  /**
   * Returns a file generated from the string.
   *
   * @param option	the current option
   * @param str		the string to convert to a file
   * @return		the generated file
   */
  public static Object valueOf(AbstractOption option, String str) {
    return new PlaceholderDirectory(str);
  }

  /**
   * Returns a custom string representation of the object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toCustomStringRepresentation(Object obj) {
    return toString(null, obj);
  }

  /**
   * Returns an object created from the custom string representation.
   *
   * @param str		the string to turn into an object
   * @return		the object
   */
  public Object fromCustomStringRepresentation(String str) {
    return valueOf(null, str);
  }

  /**
   * Returns a representation of the current property value as java source.
   *
   * @return 		a value of type 'String'
   */
  public String getJavaInitializationString() {
    File f = (File) getValue();
    if (f == null)
      return "null";
    else
      return "new adams.core.io.PlaceholderDirectory(\"" + f.getName() + "\")";
  }

  /**
   * Gets the custom editor component.
   *
   * @return 		a value of type 'Component'
   */
  protected JComponent createCustomEditor() {
    PlaceholderDirectory 	currentDir;

    currentDir = (PlaceholderDirectory) getValue();
    if (currentDir != null)
      m_DirChooser = new BaseDirectoryChooser();
    else
      m_DirChooser = new BaseDirectoryChooser(new File(System.getProperty("user.dir")));
    m_DirChooser.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	String cmdString = e.getActionCommand();
	if (cmdString.equals(BaseDirectoryChooser.APPROVE_SELECTION)) {
	  PlaceholderDirectory newVal = new PlaceholderDirectory(m_DirChooser.getSelectedFile().getAbsolutePath());
	  setValue(newVal);
	}
	closeDialog();
      }
    });

    return m_DirChooser;
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    PlaceholderDirectory 	currentFile;

    currentFile = (PlaceholderDirectory) getValue();
    if (currentFile != null)
      m_DirChooser.setSelectedFile(currentFile);
  }

  /**
   * Paints a representation of the current Object.
   *
   * @param gfx 	the graphics context to use
   * @param box 	the area we are allowed to paint into
   */
  public void paintValue(java.awt.Graphics gfx, java.awt.Rectangle box) {
    FontMetrics fm = gfx.getFontMetrics();
    int vpad = (box.height - fm.getHeight()) / 2 ;
    File f = (File) getValue();
    String val = "No directory";
    if (f != null)
      val = f.getPath();
    gfx.drawString(val, 2, fm.getHeight() + vpad);
  }
}

