/**
 * OptionsConversionPanel.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.tools;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;

import adams.core.Utils;
import adams.core.io.FileUtils;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.AbstractOptionProducer;
import adams.core.option.ArrayConsumer;
import adams.core.option.NestedProducer;
import adams.core.option.OptionHandler;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseTextAreaWithButtons;
import adams.gui.core.GUIHelper;
import adams.gui.goe.GenericObjectEditorPanel;

/**
 * Helper panel for converting options from format into another.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2491 $
 */
public class OptionsConversionPanel
  extends BasePanel {

  /** for serialization. */
  private static final long serialVersionUID = -2881786410361917678L;

  /** the GOE panel for the input. */
  protected GenericObjectEditorPanel m_InputFormat;

  /** the text area for the input. */
  protected BaseTextAreaWithButtons m_TextAreaInput;

  /** the button for copying the input. */
  protected JButton m_ButtonInputCopy;

  /** the button for pasting the input. */
  protected JButton m_ButtonInputPaste;

  /** the button for loading the input from file. */
  protected JButton m_ButtonInputOpen;

  /** the GOE panel for the output. */
  protected GenericObjectEditorPanel m_OutputFormat;

  /** the text area for the output. */
  protected BaseTextAreaWithButtons m_TextAreaOutput;

  /** the button for copying the output. */
  protected JButton m_ButtonOutputCopy;

  /** the button for pasting the output. */
  protected JButton m_ButtonOutputPaste;

  /** the button for saving the output to a file. */
  protected JButton m_ButtonOutputSave;

  /** the button initiating the conversion. */
  protected JButton m_ButtonConvert;

  /** the file chooser for loading the options. */
  protected BaseFileChooser m_FileChooserInput;

  /** the file chooser for saving the options. */
  protected BaseFileChooser m_FileChooserOutput;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_FileChooserInput  = new BaseFileChooser();
    m_FileChooserOutput = new BaseFileChooser();
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    JPanel	panelAll;
    JPanel	panel;
    JPanel	panel2;

    super.initGUI();

    setLayout(new BorderLayout());

    panelAll = new JPanel(new GridLayout(2, 1));
    add(panelAll, BorderLayout.CENTER);

    // input
    panel = new JPanel(new BorderLayout());
    panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panelAll.add(panel);

    m_InputFormat = new GenericObjectEditorPanel(AbstractOptionConsumer.class, new ArrayConsumer(), true);
    panel2 = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panel.add(panel2, BorderLayout.NORTH);
    panel2.add(m_InputFormat);

    m_TextAreaInput = new BaseTextAreaWithButtons();
    panel.add(m_TextAreaInput, BorderLayout.CENTER);

    m_ButtonInputCopy = new JButton("Copy", GUIHelper.getIcon("copy.gif"));
    m_ButtonInputCopy.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	GUIHelper.copyToClipboard(m_TextAreaInput.getText());
      }
    });
    m_TextAreaInput.addToButtonsPanel(m_ButtonInputCopy);

    m_ButtonInputPaste = new JButton("Paste", GUIHelper.getIcon("paste.gif"));
    m_ButtonInputPaste.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_TextAreaInput.setText(GUIHelper.pasteFromClipboard());
      }
    });
    m_TextAreaInput.addToButtonsPanel(m_ButtonInputPaste);

    m_ButtonInputOpen = new JButton("Open...", GUIHelper.getIcon("open.gif"));
    m_ButtonInputOpen.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int retVal = m_FileChooserInput.showOpenDialog(OptionsConversionPanel.this);
	if (retVal != BaseFileChooser.APPROVE_OPTION)
	  return;
	Vector<String> lines = FileUtils.loadFromFile(m_FileChooserInput.getSelectedFile());
	m_TextAreaInput.setText(Utils.flatten(lines, "\n"));
      }
    });
    m_TextAreaInput.addToButtonsPanel(m_ButtonInputOpen);

    // output
    panel = new JPanel(new BorderLayout());
    panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panelAll.add(panel);

    m_OutputFormat = new GenericObjectEditorPanel(AbstractOptionProducer.class, new NestedProducer(), true);
    panel2 = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panel.add(panel2, BorderLayout.NORTH);
    panel2.add(m_OutputFormat);

    m_TextAreaOutput = new BaseTextAreaWithButtons();
    panel.add(m_TextAreaOutput, BorderLayout.CENTER);

    m_ButtonOutputCopy = new JButton("Copy", GUIHelper.getIcon("copy.gif"));
    m_ButtonOutputCopy.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	GUIHelper.copyToClipboard(m_TextAreaOutput.getText());
      }
    });
    m_TextAreaOutput.addToButtonsPanel(m_ButtonOutputCopy);

    m_ButtonOutputPaste = new JButton("Paste", GUIHelper.getIcon("paste.gif"));
    m_ButtonOutputPaste.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_TextAreaOutput.setText(GUIHelper.pasteFromClipboard());
      }
    });
    m_TextAreaOutput.addToButtonsPanel(m_ButtonOutputPaste);

    m_ButtonOutputSave = new JButton("Save...", GUIHelper.getIcon("save.gif"));
    m_ButtonOutputSave.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	int retVal = m_FileChooserOutput.showSaveDialog(OptionsConversionPanel.this);
	if (retVal != BaseFileChooser.APPROVE_OPTION)
	  return;
	String filename = m_FileChooserOutput.getSelectedFile().getAbsolutePath();
	if (!FileUtils.writeToFile(filename, m_TextAreaOutput.getText(), false)) {
	  GUIHelper.showErrorMessage(
	      OptionsConversionPanel.this,
	      "Failed to write output to '" + filename + "'!");
	}
      }
    });
    m_TextAreaOutput.addToButtonsPanel(m_ButtonOutputSave);

    // conversion
    panel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    add(panel, BorderLayout.SOUTH);

    m_ButtonConvert = new JButton("Convert");
    m_ButtonConvert.setMnemonic('C');
    m_ButtonConvert.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	try {
	  AbstractOptionConsumer consumer = (AbstractOptionConsumer) m_InputFormat.getCurrent();
	  OptionHandler handler = AbstractOptionConsumer.fromString(consumer.getClass(), m_TextAreaInput.getText());
	  AbstractOptionProducer producer = (AbstractOptionProducer) m_OutputFormat.getCurrent();
	  m_TextAreaOutput.setText(AbstractOptionProducer.toString(producer.getClass(), handler));
	}
	catch (Exception ex) {
	  ex.printStackTrace();
	  GUIHelper.showErrorMessage(
	      OptionsConversionPanel.this,
	      "Failed to convert options:\n" + ex);
	}
      }
    });
    panel.add(m_ButtonConvert);
  }
}
