/*
 * LinearLeastSquares.java
 * Copyright (C) 1992 William H. Press, Saul A. Teukolsky, William T. Vetterling, Brian P. Flannery (= Numerical Recipes in C)
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand - port to Java
 */

package adams.data.fit;

import adams.core.TechnicalInformation;
import adams.core.TechnicalInformationHandler;
import adams.core.TechnicalInformation.Field;
import adams.core.TechnicalInformation.Type;
import adams.core.annotation.MixedCopyright;
import adams.core.option.OptionUtils;

/**
 <!-- globalinfo-start -->
 * Performs linear least squares to fit a function to 2-D data. Uses SVD decomposition.<br/>
 * <br/>
 * For more information, see:<br/>
 * <br/>
 * William H. Press, Saul A. Teukolsky, William T. Vetterling, Brian P. Flannery (1992). General Linear Least Squares.<br/>
 * <br/>
 * WikiPedia. Linear least squares. URL http://en.wikipedia.org/wiki/Linear_least_squares.<br/>
 * <br/>
 * WikiPedia. Singular value decomposition. URL http://en.wikipedia.org/wiki/Singular_value_decomposition.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- technical-bibtex-start -->
 * BibTeX:
 * <pre>
 * &#64;inbook{Press1992,
 *    author = {William H. Press and Saul A. Teukolsky and William T. Vetterling and Brian P. Flannery},
 *    chapter = {15.4},
 *    edition = {Second},
 *    pages = {671-681},
 *    publisher = {Cambridge University Press},
 *    series = {Numerical Recipes in C},
 *    title = {General Linear Least Squares},
 *    year = {1992},
 *    PDF = {http://www.nrbook.com/a/bookcpdf/c15-4.pdf}
 * }
 *
 * &#64;misc{missing_id,
 *    author = {WikiPedia},
 *    title = {Linear least squares},
 *    URL = {http://en.wikipedia.org/wiki/Linear_least_squares}
 * }
 *
 * &#64;misc{missing_id,
 *    author = {WikiPedia},
 *    title = {Singular value decomposition},
 *    URL = {http://en.wikipedia.org/wiki/Singular_value_decomposition}
 * }
 * </pre>
 * <p/>
 <!-- technical-bibtex-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-function &lt;adams.core.fit.LinearFunction [options]&gt; (property: function)
 *         The function to use for calculating the y values based on the x input.
 *         default: adams.core.fit.Polynomial
 * </pre>
 *
 * <pre>-iterations &lt;int&gt; (property: numIterations)
 *         The number of iterations to perform in singular value decomposition.
 *         default: 30
 * </pre>
 *
 * Default options for adams.core.fit.Polynomial (-function/function):
 *
 * <pre>-D (property: debug)
 *         If set to true, function may output additional info to the console.
 * </pre>
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2537 $
 */
@MixedCopyright
public class LinearLeastSquares
  extends Fit
  implements TechnicalInformationHandler {

  /** for serialization. */
  private static final long serialVersionUID = -1573174349202683444L;

  /** Default value for single precision and variables scaled to order unity. */
  public final static double TOL = 1.0e-5;

  /** the function to use for calculating, i.e., f(x) = y. */
  protected LinearFunction m_Function;

  /** the number of iterations to perform in svdcmp.
   * @see #svdcmp(double[][], int, int, double[], double[][], int)*/
  protected int m_NumIterations;

  /**
   * Returns a short description of the fitting method.
   *
   * @return		a description of the method
   */
  public String globalInfo() {
    return
        "Performs linear least squares to fit a function to 2-D data. "
      + "Uses SVD decomposition.\n\n"
      + "For more information, see:\n\n"
      + getTechnicalInformation().toString();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"function", "function",
	new Polynomial());

    m_OptionManager.add(
	"iterations", "numIterations",
	30);
  }

  /**
   * Returns a description of the fitting method.
   *
   * @return		a description or name
   */
  public String getDescription() {
    return "Linear least squares";
  }

  /**
   * Returns an instance of a TechnicalInformation object, containing
   * detailed information about the technical background of this class,
   * e.g., paper reference or book this class is based on.
   *
   * @return 		the technical information about this class
   */
  public TechnicalInformation getTechnicalInformation() {
    TechnicalInformation 	result;
    TechnicalInformation 	additional;

    result = new TechnicalInformation(Type.INBOOK);
    result.setValue(Field.AUTHOR, "William H. Press and Saul A. Teukolsky and William T. Vetterling and Brian P. Flannery");
    result.setValue(Field.SERIES, "Numerical Recipes in C");
    result.setValue(Field.EDITION, "Second");
    result.setValue(Field.TITLE, "General Linear Least Squares");
    result.setValue(Field.CHAPTER, "15.4");
    result.setValue(Field.PAGES, "671-681");
    result.setValue(Field.YEAR, "1992");
    result.setValue(Field.PUBLISHER, "Cambridge University Press");
    result.setValue(Field.PDF, "http://www.nrbook.com/a/bookcpdf/c15-4.pdf");

    additional = result.add(Type.MISC);
    additional.setValue(Field.AUTHOR, "WikiPedia");
    additional.setValue(Field.TITLE, "Linear least squares");
    additional.setValue(Field.URL, "http://en.wikipedia.org/wiki/Linear_least_squares");

    additional = result.add(Type.MISC);
    additional.setValue(Field.AUTHOR, "WikiPedia");
    additional.setValue(Field.TITLE, "Singular value decomposition");
    additional.setValue(Field.URL, "http://en.wikipedia.org/wiki/Singular_value_decomposition");

    return result;
  }

  /**
   * Sets the function for calculating the y values based on x, i.e., f(x) = ...
   *
   * @param value	the function to use
   */
  public void setFunction(LinearFunction value) {
    if (value != null)
      m_Function = value;
    else
      getSystemErr().println("Function cannot be null!");
  }

  /**
   * Returns the function for calculating the y values based on x, i.e., f(x) = ...
   *
   * @return		the function
   */
  public LinearFunction getFunction() {
    return m_Function;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String functionTipText() {
    return "The function to use for calculating the y values based on the x input.";
  }

  /**
   * Sets the number of iterations to perform in singular value decomposition.
   *
   * @param value	the iterations
   */
  public void setNumIterations(int value) {
    if (value > 0) {
      m_NumIterations = value;
    }
    else {
      getSystemErr().println(
	  this.getClass().getName() + ": only positive numbers are allowed for iterations!");
    }
  }

  /**
   * Returns the number of iterations to perform in singular value decomposition.
   *
   * @return		the iterations
   */
  public int getNumIterations() {
    return m_NumIterations;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String numIterationsTipText() {
    return "The number of iterations to perform in singular value decomposition.";
  }

  /**
   * Computes (a^2 + b^2)^0.5 without destructive underflow or overflow.
   *
   * @param a	the first value
   * @param b	the second value
   * @return	the result
   */
  protected double pythag(double a, double b) {
    double 	absa;
    double	absb;

    absa = Math.abs(a);
    absb = Math.abs(b);
    if (absa > absb)
      return absa * Math.sqrt(1.0 + Math.pow(absb/absa, 2));
    else
      return (absb == 0.0 ? 0.0 : absb * Math.sqrt(1.0 + Math.pow(absa/absb, 2)));
  }

  /**
   * v[0..n-1][0..n-1] as returned by svdcmp. m and n are the dimensions of a, and
   * will be equal for square matrices. b[0..m-1] is the input right-hand side.
   * x[0..n-1] is the output solution vector.  No input quantities are destroyed,
   * so the routine may be called sequentially with different b's.
   *
   * @param u		the "a" matrix
   * @param w		w as retuned by svdcmp
   * @param v		v as returned by svdcmp
   * @param m		the dimension of a/u
   * @param n		the dimension of a/u
   * @param b		the input right-hand side
   * @param x		the output solution vector
   */
  protected void svbksb(double[][] u, double w[], double[][] v, int m, int n, double b[], double x[]) {
    int 	jj;
    int		j;
    int		i;
    double 	s;
    double[]	tmp;

    tmp = new double[n];
    for (j = 0;j < n; j++) { // Calculate UT B.
      s = 0.0;
      if (w[j] != 0) { // Nonzero result only if wj is nonzero.
	for (i = 0; i < m; i++)
	  s += u[i][j] * b[i];
	s /= w[j]; // This is the divide by wj.
      }
      tmp[j] = s;
    }

    for (j = 0; j < n; j++) { // Matrix multiply by V to get answer.
      s = 0.0;
      for (jj = 0; jj < n; jj++)
	s += v[j][jj] * tmp[jj];
      x[j] = s;
    }
  }

  /**
   * Helper function, if b >=0 then abs(a) is returned otherwise -abs(a).
   *
   * @param a		first value
   * @param b		second value
   * @return		abs(a) or -abs(a)
   */
  protected double sign(double a, double b) {
    return (b) >= 0.0 ? Math.abs(a) : -Math.abs(a);
  }

  /**
   * Given a matrix a[0..m-1][0..n-1], this routine computes its singular value
   * decomposition, A = U * W * V^T .The matrix U replaces a on output. The
   * diagonal matrix of singular values W is output as a vector w[0..n-1]. The
   * matrix V (not the transpose V^T ) is output as v[0..n-1][0..n-1].
   *
   * @param a		the A matrix
   * @param m		the dimension of A
   * @param n		the dimension of A
   * @param w		the W matrix
   * @param v		the V matrix (not V^T!)
   * @param iterations	the number of iterations to perform
   * @return		true if converged
   */
  protected boolean svdcmp(double[][] a, int m, int n, double w[], double[][] v, int iterations) {
    int 	flag;
    int		i;
    int		its;
    int		j;
    int		jj;
    int		k;
    int		l;
    int		nm;
    double 	anorm;
    double	c;
    double	f;
    double	g;
    double	h;
    double	s;
    double	scale;
    double	x;
    double	y;
    double	z;
    double[]	rv1;

    rv1   = new double[n];
    g     = 0.0;
    scale = 0.0;
    anorm = 0.0; // Householder reduction to bidiagonal form.
    nm    = 0;
    l     = 0;
    for (i = 0; i < n; i++) {
      l      = i + 1;
      rv1[i] = scale * g;
      g     = 0.0;
      s     = 0.0;
      scale = 0.0;
      if (i < m) {
	for (k = i; k < m; k++)
	  scale += Math.abs(a[k][i]);
	if (scale != 1) {
	  for (k = i; k < m; k++) {
	    a[k][i] /= scale;
	    s += a[k][i]*a[k][i];
	  }

	  f       = a[i][i];
	  g       = -sign(Math.sqrt(s), f);
	  h       = f * g - s;
	  a[i][i] = f - g;
	  for (j = l; j < n; j++) {
	    for (s = 0.0, k = i; k < m; k++)
	      s += a[k][i] * a[k][j];
	    f = s / h;
	    for (k = i; k < m; k++)
	      a[k][j] += f * a[k][i];
	  }
	  for (k = i; k < m; k++)
	    a[k][i] *= scale;
	}
      }
      w[i]  = scale * g;
      g     = 0.0;
      s     = 0.0;
      scale = 0.0;
      if (i < m && (i != n - 1)) {
	for (k = l; k < n; k++)
	  scale += Math.abs(a[i][k]);
	if (scale != 1) {
	  for (k = l; k < n; k++) {
	    a[i][k] /= scale;
	    s += a[i][k] * a[i][k];
	  }

	  f       = a[i][l];
	  g       = -sign(Math.sqrt(s), f);
	  h       = f * g - s;
	  a[i][l] = f - g;
	  for (k = l; k < n; k++)
	    rv1[k] = a[i][k]/h;
	  for (j = l; j < m; j++) {
	    for (s = 0.0, k = l; k < n; k++)
	      s += a[j][k] * a[i][k];
	    for (k = l; k < n; k++)
	      a[j][k] += s * rv1[k];
	  }
	  for (k = l; k < n; k++)
	    a[i][k] *= scale;
	}
      }
      anorm = Math.max(anorm, (Math.abs(w[i])+Math.abs(rv1[i])));
    }

    for (i = n - 1; i >= 0; i--) { // Accumulation of right-hand transformations.
      if (i < n - 1) {
	if (g != 0) {
	  for (j = l; j < n; j++) // Double division to avoid possible underflow.
	    v[j][i] = (a[i][j] / a[i][l]) / g;
	  for (j = l; j < n; j++) {
	    for (s = 0.0, k = l; k < n;k++)
	      s += a[i][k] * v[k][j];
	    for (k = l; k < n; k++)
	      v[k][j] += s * v[k][i];
	  }
	}
	for (j = l; j < n;j++) {
	  v[i][j] = 0.0;
	  v[j][i] = 0.0;
	}
      }
      v[i][i] = 1.0;
      g       = rv1[i];
      l       = i;
    }

    for (i = Math.min(m,n) - 1;i >= 0; i--) { // Accumulation of left-hand transformations.
      l = i+1;
      g = w[i];
      for (j = l; j < n; j++)
	a[i][j] = 0.0;
      if (g != 0) {
	g = 1.0 / g;
	for (j = l; j < n; j++) {
	  for (s = 0.0, k = l; k < m; k++)
	    s += a[k][i] * a[k][j];
	  f = (s / a[i][i]) * g;
	  for (k = i; k < m; k++)
	    a[k][j] += f * a[k][i];
	}
	for (j = i; j < m; j++)
	  a[j][i] *= g;
      }
      else {
	for (j = i; j < m; j++)
	  a[j][i] = 0.0;
      }
      ++a[i][i];
    }

    for (k = n - 1; k >= 0; k--) { // Diagonalization of the bidiagonal form: Loop over
      for (its = 1; its <= iterations; its++) { // singular values, and over allowed iterations.
	flag = 1;
	for (l = k; l >= 0; l--) { // Test for splitting.
	  nm = l; // Note that rv1[1] is always zero.
	  if (Math.abs(rv1[l]) < 1E-6) {
	    flag = 0;
	    break;
	  }
	  if (Math.abs(w[nm]) < 1E-6) {
	    break;
	  }
	}

	if (flag == 1) {
	  c = 0.0; // Cancellation of rv1[l],if l > 1.
	  s = 1.0;
	  for (i = l; i < k; i++) {
	    f      = s * rv1[i];
	    rv1[i] = c * rv1[i];
	    if (Math.abs(f) < 1E-6)
	      break;
	    g    = w[i];
	    h    = pythag(f, g);
	    w[i] = h;
	    h    = 1.0 / h;
	    c    = g * h;
	    s    = -f * h;
	    for (j = 0; j < m; j++) {
	      y        = a[j][nm];
	      z        = a[j][i];
	      a[j][nm] = y*c + z*s;
	      a[j][i]  = z*c - y*s;
	    }
	  }
	}

	z = w[k];
	if (l == k) { // Convergence.
	  if (z < 0.0) { // Singular value is made nonnegative.
	    w[k] = -z;
	    for (j = 0; j < n; j++)
	      v[j][k] = -v[j][k];
	  }
	  break;
	}
	if (its == iterations) {
	  getSystemErr().println("no convergence in " + its + " svdcmp iterations");
	  return false;
	}

	x  = w[l]; // Shift from bottom 2-by-2 minor.
	nm = k;
	y  = w[nm];
	g  = rv1[nm];
	h  = rv1[k];
	f  = ((y-z)*(y+z) + (g-h)*(g+h)) / (2.0*h*y);
	g  = pythag(f, 1.0);
	f  = ((x-z)*(x+z) + h*((y/(f+sign(g,f)))-h)) / x;
	c  = 1.0;
	s  = 1.0; // Next QR transformation:
	for (j = l; j < nm; j++) {
	  i      = j+1;
	  g      = rv1[i];
	  y      = w[i];
	  h      = s*g;
	  g      = c*g;
	  z      = pythag(f,h);
	  rv1[j] = z;
	  c      = f/z;
	  s      = h/z;
	  f      = x*c+g*s;
	  g      = g*c-x*s;
	  h      = y*s;
	  y      *= c;
	  for (jj = 0; jj < n; jj++) {
	    x        = v[jj][j];
	    z        = v[jj][i];
	    v[jj][j] = x*c + z*s;
	    v[jj][i] = z*c - x*s;
	  }
	  z    = pythag(f,h);
	  w[j] = z; // Rotation can be arbitrary if z = 0.
	  if (z != 0) {
	    z = 1.0/z;
	    c = f*z;
	    s = h*z;
	  }
	  f = c*g + s*y;
	  x = c*y - s*g;
	  for (jj = 0; jj < m; jj++) {
	    y        = a[jj][j];
	    z        = a[jj][i];
	    a[jj][j] = y*c + z*s;
	    a[jj][i] = z*c - y*s;
	  }
	}
	rv1[l] = 0.0;
	rv1[k] = f;
	w[k]   = x;
      }
    }

    return true;
  }

  /**
   * Given a set of data points x[0..ndata-1],y[0..ndata-1] with individual standard
   * deviations sig[0..ndata-1],use chi^2 minimization to determine the coefficients
   * a[0..ma-1] of the fitting function y = SUMi ai * afunci(x). Here we solve the
   * fitting equations using singular value decomposition of the ndata by ma
   * matrix, as in section 2.6. Arrays u[0..ndata-1][0..ma-1], v[0..ma-1][0..ma-1],and w[0..ma-1]
   * provide workspace on input; on output they define the singular value
   * decomposition, and can be used to obtain the covariance matrix. The program
   * returns values for the ma fit parameters a,and chi^2 , chisq. The user supplies
   * a routine funcs(x,afunc,ma) that returns the ma basis functions evaluated at
   * x = x in the array afunc[1..ma].
   *
   * @param x		the x data points
   * @param y		the y data points
   * @param sig		the standard deviations for y
   * @param ndata	the number of data points
   * @param a		the coefficients
   * @param ma		the number of coefficients
   * @param u		helper matrix
   * @param v		helper matrix
   * @param w		helper array
   * @param chisq	array of length 1 storing chi^2
   * @param function	the function for calculating the y values based on the input x
   * @param iterations	the number of iterations to perform
   * @return		true if successfully converged
   */
  protected boolean svdfit(double x[], double y[], double sig[], int ndata, double a[], int ma, double[][] u, double[][] v, double w[], double[] chisq, LinearFunction function, int iterations) {
    boolean	result;
    int 	j;
    int		i;
    double 	wmax;
    double	tmp;
    double	thresh;
    double	sum;
    double[] 	b;
    double[] 	afunc;

    b     = new double[ndata];
    afunc = new double[ma];
    for (i = 0; i < ndata; i++) { // Accumulate coefficients of the fitting matrx
      afunc = function.calcFunctionValues(x[i], ma);
      if ((sig != null) && (sig[i] != 0.0))
	tmp = 1.0 / sig[i];
      else
	tmp = 1.0;
      for (j = 0; j < ma; j++)
	u[i][j] = afunc[j] * tmp;
      b[i] = y[i] * tmp;
    }

    result = svdcmp(u, ndata, ma, w, v, iterations); // Singular value decomposition.

    wmax = 0.0; // Edit the singular values, given TOL from the
    for (j = 0;j < ma; j++) // #define statement, between here ...
      if (w[j] > wmax)
	wmax = w[j];
    thresh = TOL*wmax;
    for (j = 0; j < ma; j++)
      if (w[j] < thresh)
	w[j] = 0.0; // ...and here.

    svbksb(u, w, v, ndata, ma, b, a);

    chisq[0] = 0.0; // Evaluate chi-square.
    for (i = 0; i < ndata; i++) {
      afunc = function.calcFunctionValues(x[i], ma);
      for (sum = 0.0, j = 0; j < ma; j++)
	sum += a[j] * afunc[j];
      if ((sig != null) && (sig[i] != 0.0))
	tmp = (y[i] - sum) / sig[i];
      else
	tmp = (y[i] - sum);
      chisq[0] += tmp * tmp;
    }

    return result;
  }

  /**
   * Use the Least Squares fit method for fitting
   * a function to 2-D data for measurements
   * y[i] vs. dependent variable x[i]. This fit assumes
   * there are errors only on the y measuresments as
   * given by the sigma_y array.<p/>
   *
   * @param parameters 	holds the coefficients
   * @param x		independent variable
   * @param y 		vertical dependent variable
   * @param sigma_x 	std. dev. error on each x value; use null to ignore array
   * @param sigma_y 	std. dev. error on each y value; use null to ignore array
   * @return		true if successfully computed
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public boolean fit(double[] parameters, double[] x, double[] y, double[] sigma_x, double[] sigma_y) {
    boolean	result;
    double[]	chisq;
    double[]	a;
    double[][]	u;
    double[][]	v;
    double[]	w;
    int		i;

    a      = new double[parameters.length];
    chisq  = new double[]{0.0};
    u      = new double[x.length][parameters.length];
    v      = new double[parameters.length][parameters.length];
    w      = new double[parameters.length];
    result = svdfit(x, y, sigma_y, x.length, a, parameters.length, u, v, w, chisq, m_Function, m_NumIterations);
    for (i = 0; i < a.length; i++)
      parameters[i] = a[i];

    return result;
  }

  /**
   * Calculates the y value based on the provided parameters and x value.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @return		the calculated y
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public double calculate(double[] parameters, double x) {
    return m_Function.calcY(x, parameters);
  }

  /**
   * Calculates the y values based on the provided parameters and x values.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @return		the calculated y's
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public double[] calculate(double[] parameters, double[] x) {
    double[]	result;
    int		i;

    result = new double[x.length];

    for (i = 0; i < x.length; i++)
      result[i] = calculate(parameters, x[i]);

    return result;
  }

  /**
   * Returns whether the initial parameters can be guessed.
   *
   * @return		true if the initial parameters can be guessed
   */
  public boolean canGuess() {
    return (m_Function instanceof InitialParameterGuesser);
  }

  /**
   * Returns the guessed initial parameters, if that is supported.
   *
   * @param x		the x values
   * @param y		the y values
   * @return		the initial parameters, null if not supported
   */
  public double[] guess(double[] x, double[] y) {
    if (canGuess())
      return ((InitialParameterGuesser) m_Function).guess(x, y);
    else
      return null;
  }

  /**
   * Returns a short description.
   *
   * @return		short description
   */
  public String toString() {
    return getDescription() + ": " + OptionUtils.getCommandLine(m_Function);
  }
}
