/**
 * Modules.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.env;

import java.io.Serializable;
import java.net.URL;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Vector;

import javax.swing.ImageIcon;

import adams.core.Properties;
import adams.gui.core.GUIHelper;

/**
 * For managing module information.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3403 $
 */
public class Modules {

  /** the name of the props file. */
  public final static String FILENAME = "Module.props";

  /** the singleton instance. */
  protected static Modules m_Singleton;

  /**
   * Container class for module information.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3403 $
   */
  public static class Module
    implements Serializable, Comparable<Module> {

    /** for serialization. */
    private static final long serialVersionUID = 7936617163709734744L;

    /** the prefix for adams modules. */
    public final static String ADAMS_PREFIX = "adams-";

    /** the name of the module. */
    protected String m_Name;

    /** the description of the module. */
    protected String m_Description;

    /** the author of the module. */
    protected String m_Author;

    /** the organization. */
    protected String m_Organization;

    /** the logo. */
    protected ImageIcon m_Logo;

    /**
     * Initializes the object.
     *
     * @param props	the module information to use
     */
    public Module(Properties props) {
      super();

      m_Name         = props.getProperty("Name", "Unknown");
      m_Description  = props.getProperty("Description", "");
      m_Author       = props.getProperty("Author", "");
      m_Organization = props.getProperty("Organization", "");
      String logo    = props.getProperty("Logo", "");
      if (logo.length() > 0) {
	try {
	  m_Logo = GUIHelper.getIcon(logo);
	  if (m_Logo == null)
	    m_Logo = new ImageIcon(ClassLoader.getSystemClassLoader().getResource(logo));
	}
	catch (Exception e) {
	  System.err.println("Failed to load image '" + logo + "' from classpath:");
	  e.printStackTrace();
	  m_Logo = GUIHelper.getIcon("unknown-module.png");
	}
      }
      else {
	m_Logo = GUIHelper.getIcon("default-module.png");
      }
    }

    /**
     * Returns the name of the module.
     *
     * @return		the name
     */
    public String getName() {
      return m_Name;
    }

    /**
     * Returns the description of the module.
     *
     * @return		the description
     */
    public String getDescription() {
      return m_Description;
    }

    /**
     * Returns the name of the author(s).
     *
     * @return		the name of the author(s)
     */
    public String getAuthor() {
      return m_Author;
    }

    /**
     * Returns the name of the organization.
     *
     * @return		the name of the organization
     */
    public String getOrganization() {
      return m_Organization;
    }

    /**
     * Returns the logo of the module.
     *
     * @return		the logo
     */
    public ImageIcon getLogo() {
      return m_Logo;
    }
    /**
     * Compares this object with the specified object for order.  Returns a
     * negative integer, zero, or a positive integer as this object is less
     * than, equal to, or greater than the specified object.
     *
     * @param   o the object to be compared.
     * @return  a negative integer, zero, or a positive integer as this object
     *		is less than, equal to, or greater than the specified object.
     */
    public int compareTo(Module o) {
      if (getName().startsWith(ADAMS_PREFIX) && o.getName().startsWith(ADAMS_PREFIX)) {
	return getName().compareTo(o.getName());
      }
      if (getName().startsWith(ADAMS_PREFIX) || o.getName().startsWith(ADAMS_PREFIX)) {
	if (getName().startsWith(ADAMS_PREFIX))
	  return -1;
	else
	  return 1;
      }
      else {
	return getName().compareTo(o.getName());
      }
    }

    /**
     * Checks whether the object is the same.
     *
     * @param o		the object to compare against
     * @return		true if the same, false otherwise
     */
    public boolean equals(Object o) {
      if (o instanceof Module)
	return (compareTo((Module) o) == 0);
      else
	return false;
    }

    /**
     * Just outputs the name.
     *
     * @return		the name
     */
    public String toString() {
      return m_Name;
    }
  }

  /** the available modules. */
  protected Vector<Module> m_Modules;

  /**
   * Initializes the module manager.
   */
  private Modules() {
    super();
    initialize();
  }

  /**
   * Initializes the modules.
   */
  protected void initialize() {
    Vector<Setup> 	setups;
    Enumeration<URL>	urls;
    URL			url;
    Properties		props;

    m_Modules = new Vector<Module>();
    setups    = Environment.getInstance().getProperties().get(ModuleDefinition.KEY);
    for (Setup setup: setups) {
      for (String dir: setup.getDirectories()) {
	try {
	  urls = ClassLoader.getSystemResources(dir + "/" + setup.getFilename());
	  while (urls.hasMoreElements()) {
	    url   = urls.nextElement();
	    props = new Properties();
	    props.load(url.openStream());
	    m_Modules.add(new Module(props));
	  }
	}
	catch (Exception e) {
	  System.err.println("Failed to list '" + dir + "/" + setup.getFilename() + "':");
	  e.printStackTrace();
	}
      }
    }
    Collections.sort(m_Modules);
  }

  /**
   * Returns the available modules.
   *
   * @return		the modules
   */
  public List<Module> getModules() {
    return m_Modules;
  }

  /**
   * Returns the singleton.
   *
   * @return		the singleton
   */
  public static synchronized Modules getSingleton() {
    if (m_Singleton == null)
      m_Singleton = new Modules();

    return m_Singleton;
  }
}
