/*
 * Sinkhole.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.HashMap;

import adams.core.Variables;
import adams.flow.core.ControlActor;
import adams.flow.core.Unknown;
import adams.flow.transformer.AbstractTransformer;
import adams.parser.BooleanExpression;
import adams.parser.GrammarSupplier;

/**
 <!-- globalinfo-start -->
 * Blocks for propagation of tokens if the expression evaluates to 'true'.<br/>
 * In case of integer or double tokens that arrive at the input, these can be accessed in the expression via 'X'.<br/>
 * <br/>
 * The following grammar is used for evaluating the boolean expressions:<br/>
 * <br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | str &lt; str<br/>
 *                | str &lt;= str<br/>
 *                | str &gt; str<br/>
 *                | str &gt;= str<br/>
 *                | str = str<br/>
 *                | str != str<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | varexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr<br/>
 *                ;<br/>
 * <br/>
 *  varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Sinkhole
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The expression to evaluate - only as long as it evaluates to 'true' the
 * &nbsp;&nbsp;&nbsp;loop actors get executed; expressions that consists solely of a variable
 * &nbsp;&nbsp;&nbsp;(eg '&#64;{blah}') get automatically wrapped in parentheses, since the expression
 * &nbsp;&nbsp;&nbsp;string gets interpreted as attached variable instead.
 * &nbsp;&nbsp;&nbsp;default: false
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3137 $
 */
public class Sinkhole
  extends AbstractTransformer
  implements ControlActor, GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -2318544907798411076L;

  /** the expression to evaluate. */
  protected String m_Expression;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Blocks for propagation of tokens if the expression evaluates to 'true'.\n"
      + "In case of integer or double tokens that arrive at the input, these "
      + "can be accessed in the expression via 'X'.\n\n"
      + "The following grammar is used for evaluating the boolean expressions:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new BooleanExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "expression", "expression",
	    "false");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("expression");

    if (variable != null)
      return variable;
    else if ((m_Expression != null) && (m_Expression.length() > 0))
      return m_Expression;
    else
      return null;
  }

  /**
   * Sets the expression to evaluate. Automatically wraps expressions in
   * parentheses that consists only of a variable. Otherwise, the expresssion
   * would get interpreted as attached variable for the expression option.
   *
   * @param value	the expression
   */
  public void setExpression(String value) {
    if (Variables.isPlaceholder(value))
      value = "(" + value + ")";
    m_Expression = value;
    reset();
  }

  /**
   * Returns the expression to evaluate.
   *
   * @return		the expression
   */
  public String getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return
        "The expression to evaluate - only as long as it evaluates to 'true' "
      + "the loop actors get executed; expressions that consists solely of a "
      + "variable (eg '@{blah}') get automatically wrapped in parentheses, "
      + "since the expression string gets interpreted as attached variable "
      + "instead.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.core.Unknown.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if ((m_Expression == null) || (m_Expression.length() == 0))
	result = "No expression provided!";
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    boolean	forward;
    String	msg;
    HashMap	symbols;
    String	exp;
    Object	value;

    result = null;

    exp     = getVariables().expand(m_Expression);
    symbols = new HashMap();
    value   = m_InputToken.getPayload();
    if (value instanceof Integer)
      symbols.put("X", ((Integer) value).doubleValue());
    else if (value instanceof Double)
      symbols.put("X", ((Double) value).doubleValue());

    try {
      forward = !BooleanExpression.evaluate(exp, symbols);
      if (getDebugLevel() >= 2)
        debug(
            "exp: " + m_Expression + "\n"
            + "  --> " + exp + "\n"
            + "  = " + (!forward) + " (" + (forward ? "" : "don't ") + "forward)", 2);
    }
    catch (Exception e) {
      forward = true;
      msg     = "Failed to evaluate expression '" + m_Expression + "': ";
      result  = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    if (forward)
      m_OutputToken = m_InputToken;
    else
      m_OutputToken = null;

    return result;
  }
}
