/*
 * AbstractGetReportValue.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.data.report.AbstractField;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.Token;

/**
 * Ancestor for transformers that retrieve a value from a report.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3528 $
 * @param <T> the type of field to handle
 */
public abstract class AbstractGetReportValue
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -6237324482439105653L;

  /** the field to get from the report. */
  protected AbstractField m_Field;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public abstract String globalInfo();

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "field", "field",
	    getDefaultField());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("field");
    if (variable != null)
      result = variable;
    else
      result = m_Field.toParseableString();

    return result;
  }

  /**
   * Returns the default field for the option.
   *
   * @return		the default field
   */
  protected abstract AbstractField getDefaultField();

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public abstract String fieldTipText();

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the accepted classes
   */
  public abstract Class[] accepts();

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Double.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    switch (m_Field.getDataType()) {
      case NUMERIC:
	return new Class[]{Double.class};
      case BOOLEAN:
	return new Class[]{Boolean.class};
      case STRING:
	return new Class[]{String.class};
      default:
	return new Class[]{String.class};
    }
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Object	obj;
    Report	report;

    result = null;
    obj    = null;

    if (m_InputToken.getPayload() instanceof ReportHandler)
      report = ((ReportHandler) m_InputToken.getPayload()).getReport();
    else
      report = (Report) m_InputToken.getPayload();

    try {
      if (report != null) {
	if (report.hasValue(m_Field)) {
	  obj = report.getValue(m_Field);
	}
	else {
	  if (isDebugOn())
	    debug("Field '" + m_Field + "' not available from report!");
	}
      }
      else {
	if (isDebugOn())
	  debug("No report available: " + m_InputToken);
      }
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    // broadcast data
    if (obj != null) {
      switch (m_Field.getDataType()) {
	case NUMERIC:
	  m_OutputToken = new Token(new Double(obj.toString()));
	  break;
	case BOOLEAN:
	  m_OutputToken = new Token(new Boolean(obj.toString()));
	  break;
	case STRING:
	  m_OutputToken = new Token(obj.toString());
	  break;
	default:
	  m_OutputToken = new Token(obj.toString());
      }
    }

    return result;
  }
}
