/*
 * ApplicationFrame.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.application;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Image;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import adams.core.PrintObject;
import adams.core.base.BasePassword;
import adams.core.base.BaseString;
import adams.core.management.Launcher;
import adams.core.management.ProcessUtils;
import adams.core.management.RestartableApplication;
import adams.core.net.InternetHelper;
import adams.core.net.ProxyHelper;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.OptionUtils;
import adams.db.AbstractDatabaseConnection;
import adams.db.AbstractIndexedTable;
import adams.db.DatabaseConnectionEstablisher;
import adams.db.DatabaseConnectionHandler;
import adams.env.Environment;
import adams.event.DatabaseConnectionChangeEvent;
import adams.event.DatabaseConnectionChangeEvent.Type;
import adams.event.DatabaseConnectionChangeListener;
import adams.gui.core.BasePanel;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.core.AbstractFrameWithOptionHandling;
import adams.gui.dialog.DatabaseConnectionsPanel;
import adams.gui.goe.AbstractEditorRegistration;
import adams.gui.scripting.ScriptingEngineHandler;
import adams.gui.scripting.ScriptingLogPanel;

/**
 * Abstract frame class for applications.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4402 $
 */
public abstract class AbstractApplicationFrame
  extends AbstractFrameWithOptionHandling
  implements DatabaseConnectionHandler, DatabaseConnectionChangeListener,
             DatabaseConnectionEstablisher, RestartableApplication {

  /** for serialization. */
  private static final long serialVersionUID = -5800519559483605870L;

  /** the props file key for the menu bar. */
  public final static String LAYOUT_MENUBAR = "MenuBar";

  /** the props file key prefix for the menus. */
  public final static String LAYOUT_MENU_PREFIX = "Menu.";

  /** the props file key for the windows menu. */
  public final static String LAYOUT_MENU_WINDOWS = "Windows";

  /** the separator between classname and shortcut. */
  public final static String LAYOUT_SHORTCUT_SEPARATOR = "#";

  /** the frame itself. */
  protected AbstractApplicationFrame m_Self;

  /** contains the child frames/windows (title &lt;-&gt; object). */
  protected HashSet<Child> m_Children = new HashSet<Child>();

  /** the "windows" menu. */
  protected JMenu m_MenuWindows;

  /** the scripting log panel. */
  protected ScriptingLogPanel m_ScriptingLogPanel;

  /** the global database connection. */
  protected AbstractDatabaseConnection m_DbConn;

  /** the title of the application. */
  protected String m_ApplicationTitle;

  /** the debugging level (0 = off, >0 = on). */
  protected int m_DebugLevel;

  /** the user mode - determines what menu entries to display. */
  protected UserMode m_UserMode;

  /** whether the application requires a reconnect of DB connection. */
  protected boolean m_RequiresReconnect;

  /** whether the application can be restarted (through Launcher class). */
  protected boolean m_EnableRestart;

  /** the menu items (classnames with further parameters) to start up immediately. */
  protected BaseString[] m_StartUps;

  /** the application menu in use. */
  protected ApplicationMenu m_AppMenu;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"D", "debugLevel",
	0, 0, null);

    m_OptionManager.add(
	"title", "applicationTitle",
	getDefaultApplicationTitle());

    m_OptionManager.add(
	"url", "URL",
	getDefaultURL(), false);

    m_OptionManager.add(
	"user", "user",
	getDefaultUser(), false);

    m_OptionManager.add(
	"password", "password",
	getDefaultPassword(), false);

    m_OptionManager.add(
	"user-mode", "userMode",
	UserMode.BASIC);

    m_OptionManager.add(
	"start-up", "startUps",
	new BaseString[0]);

    m_OptionManager.add(
	"enable-restart", "enableRestart",
	false);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    ProxyHelper.getSingleton().initializeProxy();

    m_DbConn            = getDefaultDatabaseConnection();
    m_DbConn.addChangeListener(this);
    m_RequiresReconnect = m_DbConn.getConnectOnStartUp();

    m_ScriptingLogPanel = new ScriptingLogPanel();

    m_AppMenu           = null;
  }

  /**
   * Returns the default database connection.
   *
   * @return		the default database connection
   */
  protected abstract AbstractDatabaseConnection getDefaultDatabaseConnection();

  /**
   * Returns the default title of the application.
   *
   * @return		the default title
   */
  protected abstract String getDefaultApplicationTitle();

  /**
   * Returns the database URL specified in the props file.
   *
   * @return		the default URL, if any
   */
  protected String getDefaultURL() {
    return getDefaultDatabaseConnection().getDefaultURL();
  }

  /**
   * Returns the user specified in the props file.
   *
   * @return		the default user, if any
   */
  protected String getDefaultUser() {
    return getDefaultDatabaseConnection().getDefaultUser();
  }

  /**
   * Returns the password specified in the props file.
   *
   * @return		the default password, if any
   */
  protected BasePassword getDefaultPassword() {
    return getDefaultDatabaseConnection().getDefaultPassword();
  }

  /*
   * Disconnects if necessary, due to change in connection settings.
   *
   * @see		#m_RequiresReconnect
   */
  protected void disconnect() {
    if (m_DbConn.isConnected()) {
      m_RequiresReconnect = true;
      m_DbConn.disconnect();
    }
  }

  /**
   * Sets the debugging level (0 = off).
   *
   * @param value 	>0 if debugging output should be printed
   */
  public void setDebugLevel(int value) {
    m_DebugLevel = value;
    getDatabaseConnection().setDebugLevel(value);
    getDebugging().setEnabled(value > 0);
  }

  /**
   * Returns the debugging level (0 = turned off).
   *
   * @return 		true if debugging output is on
   */
  public int getDebugLevel() {
    return m_DebugLevel;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String debugLevelTipText() {
    return "The greater the number the more additional info the scheme may output to the console (0 = off).";
  }

  /**
   * Returns true if debugging output is turned on (any level).
   *
   * @return		true if debugging output is turned on
   */
  protected boolean isDebugOn() {
    return (m_DebugLevel > 0);
  }

  /**
   * Processes the debugging message.
   *
   * @param msg		the debugging message to process
   */
  protected void debug(String msg) {
    debug(msg, 1);
  }

  /**
   * Processes the debugging message.
   *
   * @param level	the debugging level
   * @param msg		the debugging message to process
   */
  protected void debug(String msg, int level) {
    if (level <= m_DebugLevel)
      getDebugging().println(msg);
  }

  /**
   * Returns the currently set application title.
   *
   * @return		the current title
   */
  public String getApplicationTitle() {
    return m_ApplicationTitle;
  }

  /**
   * Sets the application title to use.
   *
   * @param value	the title to use
   */
  public void setApplicationTitle(String value) {
    m_ApplicationTitle = value;
    createTitle("");
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String applicationTitleTipText() {
    return "The title for the application.";
  }

  /**
   * Returns the currently set database URL.
   *
   * @return		the current URL
   */
  public String getURL() {
    return m_DbConn.getURL();
  }

  /**
   * Sets the database URL to use (disconnects if connected).
   *
   * @param value	the URL to use
   */
  public void setURL(String value) {
    if (!value.equals(getURL())) {
      disconnect();
      m_DbConn.setURL(value);
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String URLTipText() {
    return m_DbConn.URLTipText();
  }

  /**
   * Returns the currently set database user.
   *
   * @return		the current user
   */
  public String getUser() {
    return m_DbConn.getUser();
  }

  /**
   * Sets the database user to use (disconnects if connected).
   *
   * @param value	the user to use
   */
  public void setUser(String value) {
    if (!value.equals(getUser())) {
      disconnect();
      m_DbConn.setUser(value);
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String userTipText() {
    return m_DbConn.userTipText();
  }

  /**
   * Returns the currently set database password.
   *
   * @return		the current password
   */
  public BasePassword getPassword() {
    return m_DbConn.getPassword();
  }

  /**
   * Sets the database password to use (disconnects if connected).
   *
   * @param value	the password to use
   */
  public void setPassword(BasePassword value) {
    if (!value.equals(getPassword())) {
      disconnect();
      m_DbConn.setPassword(value);
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String passwordTipText() {
    return m_DbConn.passwordTipText();
  }

  /**
   * Sets the user mode - determines what menu entries are being displayed.
   *
   * @param value 	the user mode
   */
  public void setUserMode(UserMode value) {
    if (m_UserMode != value) {
      m_UserMode = value;
      setJMenuBar(createMenuBar());
    }
  }

  /**
   * Returns the current user mode - determines what menu entries are being
   * displayed.
   *
   * @return 		the user mode
   */
  public UserMode getUserMode() {
    return m_UserMode;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String userModeTipText() {
    return "The user mode, which determines the visibility of the menu items.";
  }

  /**
   * Returns the currently set menu items to immediately start up.
   *
   * @return		the menu items (incl. further parameters)
   */
  public BaseString[] getStartUps() {
    return m_StartUps;
  }

  /**
   * Sets the menu items to start up immediately.
   *
   * @param value	the menu items (incl. further parameters)
   */
  public void setStartUps(BaseString[] value) {
    m_StartUps = value;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String startUpsTipText() {
    return
        "The menu items to start up immediately; each consists of classname "
      + "and optional parameters (in case the menu definition implements "
      + AdditionalParameterHandler.class.getName() + ").";
  }

  /**
   * Sets whether to enable the restart through the Launcher.
   *
   * @param value	true if to enable restart via Launcher class
   */
  public void setEnableRestart(boolean value) {
    if (value != m_EnableRestart)  {
      m_EnableRestart = value;
      reset();
      setJMenuBar(createMenuBar());
    }
  }

  /**
   * Returns whether to enable the restart through the Launcher.
   *
   * @return		true if restart is enabled
   */
  public boolean getEnableRestart() {
    return m_EnableRestart;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String enableRestartTipText() {
    return
        "If enabled and started through the " + Launcher.class.getName()
      + " class, the application can be restarted through the menu.";
  }

  /**
   * Returns the Debugging print object.
   *
   * @return		the print object
   */
  public PrintObject getDebugging() {
    return super.getDebugging();
  }

  /**
   * Returns the SystemOut print object.
   *
   * @return		the print object
   */
  public PrintObject getSystemOut() {
    return super.getSystemOut();
  }

  /**
   * Returns the SystemErr print object.
   *
   * @return		the print object
   */
  public PrintObject getSystemErr() {
    return super.getSystemErr();
  }

  /**
   * Sets the look'n'feel.
   */
  protected void setLookAndFeel() {
    GUIHelper.setLookAndFeel(GUIHelper.getLookAndFeel());
  }

  /**
   * initializes the GUI.
   */
  protected void initGUI() {
    super.initGUI();

    m_Self = this;

    setLookAndFeel();

    createTitle("");
    setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);

    setJMenuBar(createMenuBar());

    // size + position
    pack();
    setSize(getGraphicsConfiguration().getBounds().width, getHeight());
    setLocation(0, 0);
  }

  /**
   * finishes the initialization, by setting size/location.
   */
  protected void finishInit() {
    super.finishInit();
    reconnectIfRequired();
    // initialize database connections
    new DatabaseConnectionsPanel();
    createTitle("");
    AbstractEditorRegistration.registerEditors();
  }

  /**
   * Closes all children frames.
   */
  protected void closeChildren() {
    Iterator<Child> 	iter;
    Vector<Child> 	list;
    int 		i;
    Child 		c;

    // close all children
    iter = getWindowList();
    list = new Vector<Child>();
    while (iter.hasNext())
      list.add(iter.next());
    for (i = 0; i < list.size(); i++) {
      c = list.get(i);
      c.dispose();
    }
  }

  /**
   * Closes the application.
   */
  protected void closeApplication() {
    m_Self.dispose();
  }

  /**
   * Closes the application down.
   */
  public void close() {
    // close all children
    closeChildren();

    // close main window
    closeApplication();

    // make sure we stop
    System.exit(0);
  }

  /**
   * Returns the filename that stores the menu layout.
   *
   * @return		the filename
   */
  protected abstract String getMenuLayoutFile();

  /**
   * Returns the menu generator.
   *
   * @return		the menu generator
   */
  protected synchronized ApplicationMenu getAppMenu() {
    if (m_AppMenu == null) {
      m_AppMenu = new ApplicationMenu(this);
      m_AppMenu.setSetup(getMenuLayoutFile());
      m_AppMenu.setUserMode(getUserMode());
    }

    return m_AppMenu;
  }

  /**
   * Creates the menu bar.
   *
   * @return		the generated menu bar
   */
  protected JMenuBar createMenuBar() {
    JMenuBar		result;

    m_AppMenu     = null;
    result        = getAppMenu().getMenuBar();
    m_MenuWindows = getAppMenu().getWindowsMenu();

    return result;
  }

  /**
   * creates a frame and returns it.
   *
   * @param title		the title of the frame
   * @param c			the component to place, can be null
   * @param width		the width of the frame, ignored if -1
   * @param height		the height of the frame, ignored if -1
   * @param icon		the icon to use, null for default
   * @return			the generated frame
   */
  protected ChildFrame createChildFrame(String title, Component c, int width, int height, String icon) {
    ChildFrame 			result;
    int 			screenHeight;
    int 			screenWidth;
    ScriptingEngineHandler 	handler;

    result = new ChildFrame(this, title, icon);

    // layout
    result.setLayout(new BorderLayout());
    if (c != null)
      result.getContentPane().add(c, BorderLayout.CENTER);

    // size
    result.pack();
    if ((width > -1) && (height > -1))
      result.setSize(width, height);
    result.validate();

    // location
    screenHeight = getGraphicsConfiguration().getBounds().height;
    screenWidth  = getGraphicsConfiguration().getBounds().width;
    result.setLocation(
	(screenWidth - result.getBounds().width) / 2,
	(screenHeight - result.getBounds().height) / 2);

    // custom size and location
    if (c != null)
      GUIHelper.setSizeAndLocation(result, c);

    // add listener
    result.addDisposeWindowListener();

    // menu bar?
    if ((c != null) && (c instanceof MenuBarProvider))
      result.setJMenuBar(((MenuBarProvider) c).getMenuBar());

    // startup script?
    if ((c != null) && (c instanceof ScriptingEngineHandler) && (c instanceof BasePanel)) {
      handler = (ScriptingEngineHandler) c;
      if (GUIHelper.getStartupScript(c) != null)
	handler.getScriptingEngine().add((BasePanel) c, GUIHelper.getStartupScript(c));
    }

    // display frame
    result.setVisible(true);

    return result;
  }

  /**
   * creates a window and returns it.
   *
   * @param title		the title of the frame
   * @param c			the component to place, can be null
   * @param width		the width of the frame, ignored if -1
   * @param height		the height of the frame, ignored if -1
   * @param icon		the icon to use, null for default
   * @return			the generated frame
   */
  protected ChildWindow createChildWindow(String title, Component c, int width, int height, String icon) {
    ChildWindow 		result;
    int 			screenHeight;
    int 			screenWidth;
    ScriptingEngineHandler 	handler;

    result = new ChildWindow(this, title, icon);

    // layout
    result.setLayout(new BorderLayout());
    if (c != null) {
      result.getContentPane().add(c, BorderLayout.CENTER);
      if ((this instanceof DatabaseConnectionHandler) && (c instanceof DatabaseConnectionHandler))
	((DatabaseConnectionHandler) c).setDatabaseConnection(
	    ((DatabaseConnectionHandler) this).getDatabaseConnection());
    }

    // size
    result.pack();
    if ((width > -1) && (height > -1))
      result.setSize(width, height);
    result.validate();

    // location
    screenHeight = getGraphicsConfiguration().getBounds().height;
    screenWidth  = getGraphicsConfiguration().getBounds().width;
    result.setLocation(
	(screenWidth - result.getBounds().width) / 2,
	(screenHeight - result.getBounds().height) / 2);

    // custom size and location
    if (c != null)
      GUIHelper.setSizeAndLocation(result, c);

    // add listener
    result.addDisposeWindowListener();

    // startup script?
    if ((c != null) && (c instanceof ScriptingEngineHandler) && (c instanceof BasePanel)) {
      handler = (ScriptingEngineHandler) c;
      if (GUIHelper.getStartupScript(c) != null)
	handler.getScriptingEngine().add((BasePanel) c, GUIHelper.getStartupScript(c));
    }

    // display frame
    result.setVisible(true);

    return result;
  }

  /**
   * insert the menu item in a sorted fashion.
   *
   * @param menu	the menu to add the item to
   * @param menuitem	the menu item to add
   */
  protected void insertMenuItem(JMenu menu, JMenuItem menuitem) {
    insertMenuItem(menu, menuitem, 0);
  }

  /**
   * insert the menu item in a sorted fashion.
   *
   * @param menu	the menu to add the item to
   * @param menuitem	the menu item to add
   * @param startIndex	the index in the menu to start with (0-based)
   */
  protected void insertMenuItem(JMenu menu, JMenuItem menuitem, int startIndex) {
    boolean	inserted;
    int		i;
    JMenuItem	current;
    String	currentStr;
    String	newStr;

    inserted = false;
    newStr   = menuitem.getText().toLowerCase();

    // try to find a spot inbetween
    for (i = startIndex; i < menu.getMenuComponentCount(); i++) {
      if (!(menu.getMenuComponent(i) instanceof JMenuItem))
	continue;

      current    = (JMenuItem) menu.getMenuComponent(i);
      currentStr = current.getText().toLowerCase();
      if (currentStr.compareTo(newStr) > 0) {
	inserted = true;
	menu.insert(menuitem, i);
	break;
      }
    }

    // add it at the end if not yet inserted
    if (!inserted)
      menu.add(menuitem);
  }

  /**
   * creates and displays the title.
   *
   * @param title 	the additional part of the title
   */
  public void createTitle(String title) {
    String				newTitle;
    String				name;
    HashSet<AbstractDatabaseConnection>	conns;
    HashSet<String>			connsStr;
    Vector<String>			connsList;

    newTitle = getApplicationTitle();
    name     = InternetHelper.getLocalHostName();
    if (name != null)
      newTitle += "@" + name;

    conns = AbstractDatabaseConnection.getActiveConnectionObjects();
    if (conns.size() > 0) {
      connsStr = new HashSet<String>();
      for (AbstractDatabaseConnection conn: conns)
	connsStr.add(conn.toStringShort());
      connsList = new Vector<String>(connsStr);
      Collections.sort(connsList);
      newTitle += " " + connsList;
    }
    else {
      newTitle += " [-not connected-]";
    }

    if (title.length() != 0) {
      if (title.length() > 50)
	newTitle += " - " + title.substring(0, 50) + "...";
      else
	newTitle += " - " + title;
    }

    setTitle(newTitle);
  }

  /**
   * adds the given child frame to the list of frames.
   *
   * @param c 		the child frame to add
   */
  public void addChildFrame(ChildFrame c) {
    m_Children.add(c);
    windowListChanged();
  }

  /**
   * adds the given child frame to the list of frames.
   *
   * @param c 		the child frame to add
   */
  public void addChildWindow(ChildWindow c) {
    m_Children.add(c);
    windowListChanged();
  }

  /**
   * tries to remove the child frame, it returns true if it could do such.
   *
   * @param c 		the child frame to remove
   * @return 		true if the child frame could be removed
   */
  public boolean removeChildFrame(Container c) {
    boolean result = m_Children.remove(c);
    windowListChanged();
    return result;
  }

  /**
   * brings child frame to the top.
   *
   * @param c 		the frame to activate
   * @return 		true if frame was activated
   */
  public boolean showWindow(Child c) {
    boolean	result;

    result = false;

    if (c != null) {
      createTitle(c.getTitle());
      if (c instanceof ChildFrame)
	((ChildFrame) c).setExtendedState(JFrame.NORMAL);
      c.toFront();
      c.requestFocus();
      result = true;
    }

    return result;
  }

  /**
   * brings the first frame to the top that is of the specified
   * window class.
   *
   * @param windowClass	the class to display the first child for
   * @return		true, if a child was found and brought to front
   */
  public boolean showWindow(Class windowClass) {
    return showWindow(getWindow(windowClass));
  }

  /**
   * returns all currently open frames.
   *
   * @return 		an iterator over all currently open frame
   */
  public Iterator<Child> getWindowList() {
    return m_Children.iterator();
  }

  /**
   * returns the first instance of the given window class, null if none can be
   * found.
   *
   * @param windowClass	the class to retrieve the first instance for
   * @return		null, if no instance can be found
   */
  public Child getWindow(Class windowClass) {
    Child		result;
    Iterator<Child>	iter;
    Child		current;

    result = null;
    iter   = getWindowList();
    while (iter.hasNext()) {
      current = iter.next();
      if (current.getClass() == windowClass) {
        result = current;
        break;
      }
    }

    return result;
  }

  /**
   * returns the first window with the given title, null if none can be
   * found.
   *
   * @param title	the title to look for
   * @return		null, if no instance can be found
   */
  public Child getWindow(String title) {
    Child		result;
    Iterator<Child>	iter;
    Child		current;
    boolean		found;

    result = null;
    iter   = getWindowList();
    while (iter.hasNext()) {
      current = iter.next();
      found   = current.getTitle().equals(title);

      if (found) {
        result = current;
        break;
      }
    }

    return result;
  }

  /**
   * checks, whether an instance of the given window class is already in
   * the Window list.
   *
   * @param windowClass	the class to check for an instance in the current
   * 			window list
   * @return		true if the class is already listed in the Window list
   */
  public boolean containsWindow(Class windowClass) {
    return (getWindow(windowClass) != null);
  }

  /**
   * checks, whether a window with the given title is already in
   * the Window list.
   *
   * @param title	the title to check for in the current window list
   * @return		true if a window with the given title is already
   * 			listed in the Window list
   */
  public boolean containsWindow(String title) {
    return (getWindow(title) != null);
  }

  /**
   * minimizes all windows.
   */
  public void minimizeWindows() {
    Iterator<Child>	iter;
    Child		child;

    iter = getWindowList();
    while (iter.hasNext()) {
      child = iter.next();
      try {
	if (child instanceof ChildFrame)
	  ((ChildFrame) child).setExtendedState(JFrame.ICONIFIED);
      }
      catch (Exception e) {
	e.printStackTrace();
      }
    }
  }

  /**
   * restores all windows.
   */
  public void restoreWindows() {
    Iterator<Child>	iter;
    Child		child;

    iter = getWindowList();
    while (iter.hasNext()) {
      child = iter.next();
      try {
	if (child instanceof ChildFrame)
	  ((ChildFrame) child).setExtendedState(JFrame.NORMAL);
    }
      catch (Exception e) {
	e.printStackTrace();
      }
    }
  }

  /**
   * is called when window list changed somehow (add or remove).
   */
  public void windowListChanged() {
    buildWindowsMenu();
  }

  /**
   * creates the menu of currently open windows.
   */
  protected synchronized void buildWindowsMenu() {
    Iterator<Child>	iter;
    JMenuItem		menuitem;
    int			startIndex;
    List<Image>		images;
    boolean		useEmpty;

    // remove all existing entries
    m_MenuWindows.removeAll();

    // minimize + restore + separator
    menuitem = new JMenuItem("Minimize");
    menuitem.setIcon(GUIHelper.getIcon("minimize.png"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        minimizeWindows();
      }
    });
    m_MenuWindows.add(menuitem);

    menuitem = new JMenuItem("Restore");
    menuitem.setIcon(GUIHelper.getIcon("maximize.png"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent evt) {
        restoreWindows();
      }
    });
    m_MenuWindows.add(menuitem);

    m_MenuWindows.addSeparator();

    // windows
    startIndex = m_MenuWindows.getMenuComponentCount() - 1;
    iter       = getWindowList();
    m_MenuWindows.setVisible(iter.hasNext());
    while (iter.hasNext()) {
      Child child = iter.next();
      menuitem = new JMenuItem(child.getTitle());
      useEmpty = true;
      if (child instanceof Window) {
	images = ((Window) child).getIconImages();
	if (images.size() > 0) {
	  useEmpty = false;
	  menuitem.setIcon(new ImageIcon(images.get(0)));
	}
      }
      if (useEmpty)
	menuitem.setIcon(GUIHelper.getEmptyIcon());
      insertMenuItem(m_MenuWindows, menuitem, startIndex);
      menuitem.setActionCommand(Integer.toString(child.hashCode()));
      menuitem.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent evt) {
          Iterator<Child> iter = getWindowList();
          while (iter.hasNext()) {
            Child child = iter.next();
            String hashFrame = Integer.toString(child.hashCode());
            if (hashFrame.equals(evt.getActionCommand())) {
              showWindow(child);
              break;
            }
          }
        }
      });
    }
  }

  /**
   * Shows or hides this component depending on the value of parameter b.
   *
   * @param b		if true, shows this component; otherwise, hides this
   * 			component
   */
  public void setVisible(boolean b) {
    super.setVisible(b);

    if (b)
      repaint();
  }

  /**
   * Returns the currently used database connection object, can be null.
   *
   * @return		the current object
   */
  public AbstractDatabaseConnection getDatabaseConnection() {
    return m_DbConn;
  }

  /**
   * Sets the database connection object to use.
   *
   * @param value	the object to use
   */
  public void setDatabaseConnection(AbstractDatabaseConnection value) {
    m_DbConn = value;
  }

  /**
   * A change in the database connection occurred.
   *
   * @param e		the event
   */
  public void databaseConnectionStateChanged(DatabaseConnectionChangeEvent e) {
    createTitle("");
    if (e.getType() == Type.CONNECT)
      m_DbConn = e.getDatabaseConnection();
  }

  /**
   * Re-establishes the database connection, if necessary.
   *
   * @see		#m_RequiresReconnect
   */
  protected void reconnectIfRequired() {
    if (m_RequiresReconnect) {
      m_RequiresReconnect = false;
      try {
	m_DbConn.connect();
      }
      catch (Exception e) {
	getSystemErr().println("Failed to reconnect to database:");
	getSystemErr().printStackTrace(e);
      }
    }
  }

  /**
   * Establishes the database connection.
   *
   * @see		#reconnectIfRequired()
   */
  public void establishDatabaseConnection() {
    reconnectIfRequired();
  }

  /**
   * Starts up any menu items that were defined.
   *
   * @see		#m_StartUps
   */
  protected void startUpMenuItems() {
    int						i;
    AbstractMenuItemDefinition			item;
    final Vector<AbstractMenuItemDefinition>	items;
    Runnable					runnable;

    // collect menu items
    items = new Vector<AbstractMenuItemDefinition>();
    for (i = 0; i < m_StartUps.length; i++) {
      item = AbstractMenuItemDefinition.forCommandLine(this, m_StartUps[i].toString());
      if (getAppMenu().isBlacklisted(item.getClass())) {
	getSystemErr().println(
	    item.getClass() + " is blacklisted and cannot be displayed!");
	continue;
      }
      if (getUserMode().compareTo(item.getUserMode()) < 0) {
	getSystemErr().println(
	    item.getClass() + " requires at least user mode '"
	    + getAppMenu().getUserMode() + "' (current: '" + getUserMode() + "')!");
	continue;
      }
      items.add(item);
    }

    // start up menu items
    if (items.size() > 0) {
      runnable = new Runnable() {
	public void run() {
	  for (int i = 0; i < items.size(); i++)
	    items.get(i).launch();
	}
      };
      SwingUtilities.invokeLater(runnable);
    }
  }

  /**
   * Returns the scripting log panel instance.
   *
   * @return		the panel
   */
  public ScriptingLogPanel getScriptingLogPanel() {
    return m_ScriptingLogPanel;
  }

  /**
   * Runs the application from the commandline.
   *
   * @param env		the environment class to use
   * @param app		the application frame class
   * @param options	the commandline options
   * @return		the instantiated frame, null in case of an error or
   * 			invocation of help
   */
  public static AbstractApplicationFrame runApplication(Class env, Class app, String[] options) {
    AbstractApplicationFrame	result;

    Environment.setEnvironmentClass(env);

    try {
      if (OptionUtils.helpRequested(options)) {
	System.out.println("Help requested...\n");
	result = forName(app.getName(), new String[0]);
	System.out.println("\n" + OptionUtils.list(result));
	result.dispose();
	result = null;
      }
      else {
	result = forName(app.getName(), options);
	Environment.getInstance().setApplicationFrame(result);
	if (result.getDatabaseConnection().isConnected())
	  AbstractIndexedTable.initTables(result.getDatabaseConnection());
	result.getSystemOut().println("PID: " + ProcessUtils.getVirtualMachinePID());
	result.setVisible(true);
	result.startUpMenuItems();
      }
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the application frame with the given options.
   *
   * @param classname	the classname of the application frame to instantiate
   * @param options	the options for the application frame
   * @return		the instantiated application frame or null if an error occurred
   */
  public static AbstractApplicationFrame forName(String classname, String[] options) {
    AbstractApplicationFrame	result;

    try {
      result = (AbstractApplicationFrame) OptionUtils.forName(AbstractApplicationFrame.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the application frame from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			application frame to instantiate
   * @return		the instantiated application frame
   * 			or null if an error occurred
   */
  public static AbstractApplicationFrame forCommandLine(String cmdline) {
    return (AbstractApplicationFrame) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
