/**
 * ClipboardActorContainer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.flow.tree;

import java.util.ArrayList;
import java.util.Arrays;

import adams.core.option.AbstractOptionProducer;
import adams.core.option.NestedConsumer;
import adams.core.option.NestedProducer;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.MutableActorHandler;

/**
 * A simple container for multiple actors, for easy copying to and retrieving
 * from the clipboard.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3984 $
 */
public class ClipboardActorContainer
  extends AbstractActor
  implements MutableActorHandler {

  /** for serialization. */
  private static final long serialVersionUID = 8826708859698352085L;

  /** the actors. */
  protected ArrayList<AbstractActor> m_Actors;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Container for copying actors to and pasting from the clipboard";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "actor", "actors",
	    new AbstractActor[0]);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Actors = new ArrayList<AbstractActor>();
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(true, ActorExecution.UNDEFINED, false);
  }

  /**
   * Performs checks on the "sub-actors".
   *
   * @return		always null
   */
  public String check() {
    return null;
  }

  /**
   * Returns the size of the group.
   *
   * @return		the size
   */
  public int size() {
    return m_Actors.size();
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public AbstractActor get(int index) {
    return m_Actors.get(index);
  }

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public void set(int index, AbstractActor actor) {
    m_Actors.set(index, actor);
    reset();
  }

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public int indexOf(String actor) {
    int		result;
    int		i;

    result = -1;
    for (i = 0; i < m_Actors.size(); i++) {
      if (m_Actors.get(i).getName().equals(actor)) {
	result = i;
	break;
      }
    }

    return result;
  }

  /**
   * Returns the first non-skipped actor.
   *
   * @return		the first 'active' actor, null if none available
   */
  public AbstractActor firstActive() {
    AbstractActor	result;
    int			i;

    result = null;

    for (i = 0; i < m_Actors.size(); i++) {
      if (!m_Actors.get(i).getSkip()) {
	result = m_Actors.get(i);
	break;
      }
    }

    return result;
  }

  /**
   * Returns the last non-skipped actor.
   *
   * @return		the last 'active' actor, null if none available
   */
  public AbstractActor lastActive() {
    AbstractActor	result;
    int			i;

    result = null;

    for (i = m_Actors.size() - 1; i >= 0; i--) {
      if (!m_Actors.get(i).getSkip()) {
	result = m_Actors.get(i);
	break;
      }
    }

    return result;
  }

  /**
   * Inserts the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to insert
   */
  public void add(int index, AbstractActor actor) {
    m_Actors.add(index, actor);
    reset();
  }

  /**
   * Removes the actor at the given position and returns the removed object.
   *
   * @param index	the position
   * @return		the removed actor
   */
  public AbstractActor remove(int index) {
    AbstractActor	result;

    result = m_Actors.remove(index);
    reset();

    return result;
  }

  /**
   * Removes all actors.
   */
  public void removeAll() {
    m_Actors.clear();
    reset();
  }

  /**
   * Sets the actors. Replaces all previous actors.
   *
   * @param value	the actors to set
   */
  public void setActors(AbstractActor[] value) {
    m_Actors.clear();
    m_Actors.addAll(Arrays.asList(value));
    reset();
  }

  /**
   * Returns the actors as array.
   *
   * @return		the actors
   */
  public AbstractActor[] getActors() {
    return m_Actors.toArray(new AbstractActor[m_Actors.size()]);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorsTipText() {
    return "The actors.";
  }

  /**
   * Does nothing.
   *
   * @return		always null
   */
  protected String doExecute() {
    return null;
  }

  /**
   * Returns a string representation of the options.
   *
   * @return		 a string representation
   */
  public String toNestedString() {
    return AbstractOptionProducer.toString(NestedProducer.class, this);
  }

  /**
   * Creates, if possible, an actor container from the given string (in nested
   * format).
   *
   * @param s		the string parse
   * @return		the parsed actor or null if not possible
   */
  public static ClipboardActorContainer fromNestedString(String s) {
    ClipboardActorContainer	result;
    Object			obj;
    NestedConsumer		consumer;

    consumer = new NestedConsumer();
    consumer.setQuiet(true);
    obj = consumer.fromString(s);
    consumer.cleanUp();

    result = null;
    if ((obj != null) && (obj instanceof ClipboardActorContainer)) {
      result = (ClipboardActorContainer) obj;
    }
    else if (obj instanceof AbstractActor) {
      result = new ClipboardActorContainer();
      result.setActors(new AbstractActor[]{(AbstractActor) obj});
    }

    return result;
  }
}
