/**
 * AbstractPropertyEditor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.awt.Graphics;
import java.awt.Rectangle;

import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

/**
 * A superclass for custom editor for basic Java types.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 1733 $
 */
public abstract class AbstractBasicTypePropertyEditor
  extends AbstractPropertyEditorSupport {

  /**
   * Determines whether this property editor is paintable.
   *
   * @return  True if the class will honor the paintValue method.
   */

  public boolean isPaintable() {
    return false;
  }

  /**
   * Paint a representation of the value into a given area of screen
   * real estate.  Note that the propertyEditor is responsible for doing
   * its own clipping so that it fits into the given rectangle.
   * <p>
   * If the PropertyEditor doesn't honor paint requests (see isPaintable)
   * this method should be a silent noop.
   * <p>
   * The given Graphics object will have the default font, color, etc of
   * the parent container.  The PropertyEditor may change graphics attributes
   * such as font and color and doesn't need to restore the old values.
   *
   * @param gfx  Graphics object to paint into.
   * @param box  Rectangle within graphics object into which we should paint.
   */
  public void paintValue(Graphics gfx, Rectangle box) {
    // does nothing
  }

  /**
   * Gets the property value as text.
   *
   * @return The property value as a human editable string.
   * <p>   Returns null if the value can't be expressed as an editable string.
   * <p>   If a non-null value is returned, then the PropertyEditor should
   *	     be prepared to parse that string back in setAsText().
   */
  public String getAsText() {
    return getValue().toString();
  }

  /**
   * Set the property value by parsing a given String.  May raise
   * java.lang.IllegalArgumentException if either the String is
   * badly formatted or if this kind of property can't be expressed
   * as text.
   *
   * @param text  	The string to be parsed.
   * @throws IllegalArgumentException 	if parsing fails
   * @see 		#parse(String)
   */
  public void setAsText(String text) throws IllegalArgumentException {
    setValue(parse(text));
  }

  /**
   * Parses the string and returns an object of the correct class.
   *
   * @param text	the string to parse
   * @return		the generated object
   * @throws IllegalArgumentException	if parsing fails
   */
  protected abstract Object parse(String text) throws IllegalArgumentException;

  /**
   * Creates the custom editor to use.
   *
   * @return		the custom editor
   */
  protected JComponent createCustomEditor() {
    JTextField	result;

    result = new JTextField();
    result.getDocument().addDocumentListener(new DocumentListener() {
      public void removeUpdate(DocumentEvent e) {
	set(e.getDocument());
      }
      public void insertUpdate(DocumentEvent e) {
	set(e.getDocument());
      }
      public void changedUpdate(DocumentEvent e) {
	set(e.getDocument());
      }
      protected void set(Document doc) {
	try {
	  Object value = parse(doc.getText(0, doc.getLength()));
	  if (!value.equals(getValue()))
	    setValue(value);
	}
	catch (Exception e) {
	  e.printStackTrace();
	}
      }
    });

    return result;
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    if (!((JTextField) m_CustomEditor).getText().equals(getValue().toString()))
      ((JTextField) m_CustomEditor).setText(getValue().toString());
  }
}
