/*
 * Utils.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 * Copyright (C) 2006 by Dr. Herong Yang, http://www.herongyang.com/
 */

package adams.core;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.lang.reflect.Array;
import java.text.BreakIterator;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import adams.core.annotation.MixedCopyright;

/**
 * Class implementing some simple utility methods.
 *
 * @author Eibe Frank
 * @author Yong Wang
 * @author Len Trigg
 * @author Julien Prados
 * @author FracPete (fracpete at waikat dot ac dot nz)
 * @author Java-Tips.org
 * @author Herong Yang
 * @version $Revision: 4436 $
 * @see weka.core.Utils
 */
@MixedCopyright
public class Utils {

  /** hexadecimal digits. */
  public static char HEX_DIGIT[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

  /**
   * Rounds a double and converts it into String. Always displays the
   * specified number of decimals.
   *
   * @param value 		the double value
   * @param afterDecimalPoint 	the number of digits permitted
   * 				after the decimal point; if -1 then all
   * 				decimals are displayed; also if number > Long.MAX_VALUE
   * @return 			the double as a formatted string
   */
  public static String doubleToStringFixed(double value, int afterDecimalPoint) {
    StringBuilder	result;
    double		valueNew;
    double		factor;
    StringBuilder	remainder;

    if (    (afterDecimalPoint < 0)
	 || (value > Long.MAX_VALUE)
	 || (value < Long.MIN_VALUE) ) {
      result = new StringBuilder(Double.toString(value));
    }
    else {
      factor    = Math.pow(10, afterDecimalPoint);
      valueNew  = Math.floor(value * factor) / factor;
      result    = new StringBuilder(Long.toString(Math.round(Math.floor(valueNew))));
      remainder = new StringBuilder("" + (long) Math.round((valueNew - Math.floor(valueNew)) * Math.pow(10, afterDecimalPoint)));
      remainder.delete(0, remainder.indexOf(".") + 1);
      if (afterDecimalPoint > 0) {
	while (remainder.length() < afterDecimalPoint)
	  remainder.append('0');
	result.append('.');
	result.append(remainder.substring(0, afterDecimalPoint));
      }
    }

    return result.toString();
  }

  /**
   * Rounds a double and converts it into String.
   *
   * @param value 		the double value
   * @param afterDecimalPoint 	the (maximum) number of digits permitted
   * 				after the decimal point
   * @return 			the double as a formatted string
   */
  public static String doubleToString(double value, int afterDecimalPoint) {

    StringBuffer stringBuffer;
    double temp;
    int dotPosition;
    long precisionValue;

    temp = value * Math.pow(10.0, afterDecimalPoint);
    if (Math.abs(temp) < Long.MAX_VALUE) {
      precisionValue = 	(temp > 0) ? (long)(temp + 0.5)
	  : -(long)(Math.abs(temp) + 0.5);
      if (precisionValue == 0)
	stringBuffer = new StringBuffer(String.valueOf(0));
      else
	stringBuffer = new StringBuffer(String.valueOf(precisionValue));

      if (afterDecimalPoint == 0)
	return stringBuffer.toString();

      dotPosition = stringBuffer.length() - afterDecimalPoint;
      while (((precisionValue < 0) && (dotPosition < 1)) || (dotPosition < 0)) {
	if (precisionValue < 0)
	  stringBuffer.insert(1, '0');
	else
	  stringBuffer.insert(0, '0');
	dotPosition++;
      }

      stringBuffer.insert(dotPosition, '.');

      if ((precisionValue < 0) && (stringBuffer.charAt(1) == '.'))
	stringBuffer.insert(1, '0');
      else if (stringBuffer.charAt(0) == '.')
	stringBuffer.insert(0, '0');

      int currentPos = stringBuffer.length() - 1;
      while ((currentPos > dotPosition) && (stringBuffer.charAt(currentPos) == '0'))
	stringBuffer.setCharAt(currentPos--, ' ');

      if (stringBuffer.charAt(currentPos) == '.')
	stringBuffer.setCharAt(currentPos, ' ');

      return stringBuffer.toString().trim();
    }
    return new String("" + value);
  }

  /**
   * Rounds a double and converts it into a formatted decimal-justified String.
   * Trailing 0's are replaced with spaces.
   *
   * @param value 		the double value
   * @param width 		the width of the string
   * @param afterDecimalPoint 	the number of digits after the decimal point
   * @return 			the double as a formatted string
   */
  public static String doubleToString(double value, int width, int afterDecimalPoint) {

    String tempString = doubleToString(value, afterDecimalPoint);
    char[] result;
    int dotPosition;

    if ((afterDecimalPoint >= width)
	|| (tempString.indexOf('E') != -1)) { // Protects sci notation
      return tempString;
    }

    // Initialize result
    result = new char[width];
    for (int i = 0; i < result.length; i++)
      result[i] = ' ';

    if (afterDecimalPoint > 0) {
      // Get position of decimal point and insert decimal point
      dotPosition = tempString.indexOf('.');
      if (dotPosition == -1)
	dotPosition = tempString.length();
      else
	result[width - afterDecimalPoint - 1] = '.';
    }
    else {
      dotPosition = tempString.length();
    }

    int offset = width - afterDecimalPoint - dotPosition;
    if (afterDecimalPoint > 0)
      offset--;

    // Not enough room to decimal align within the supplied width
    if (offset < 0) {
      return tempString;
    }

    // Copy characters before decimal point
    for (int i = 0; i < dotPosition; i++)
      result[offset + i] = tempString.charAt(i);

    // Copy characters after decimal point
    for (int i = dotPosition + 1; i < tempString.length(); i++)
      result[offset + i] = tempString.charAt(i);

    return new String(result);
  }

  /**
   * Returns the basic class of an array class (handles multi-dimensional
   * arrays).
   * @param c        the array to inspect
   * @return         the class of the innermost elements
   */
  public static Class getArrayClass(Class c) {
    if (c.getComponentType().isArray())
      return getArrayClass(c.getComponentType());
    else
      return c.getComponentType();
  }

  /**
   * Returns the dimensions of the given array. Even though the
   * parameter is of type "Object" one can hand over primitve arrays, e.g.
   * int[3] or double[2][4].
   *
   * @param array       the array to determine the dimensions for
   * @return            the dimensions of the array
   */
  public static int getArrayDimensions(Class array) {
    if (array.getComponentType().isArray())
      return 1 + getArrayDimensions(array.getComponentType());
    else
      return 1;
  }

  /**
   * Returns the dimensions of the given array. Even though the
   * parameter is of type "Object" one can hand over primitve arrays, e.g.
   * int[3] or double[2][4].
   *
   * @param array       the array to determine the dimensions for
   * @return            the dimensions of the array
   */
  public static int getArrayDimensions(Object array) {
    return getArrayDimensions(array.getClass());
  }

  /**
   * Returns the given Array in a string representation. Even though the
   * parameter is of type "Object" one can hand over primitve arrays, e.g.
   * int[3] or double[2][4].
   *
   * @param array       the array to return in a string representation
   * @param outputClass	whether to output the class name instead of calling
   * 			the object's "toString()" method
   * @return            the array as string
   */
  public static String arrayToString(Object array, boolean outputClass) {
    StringBuilder	result;
    int			dimensions;
    int			i;
    Object		obj;

    result     = new StringBuilder();
    dimensions = getArrayDimensions(array);

    if (dimensions == 0) {
      result.append("null");
    }
    else if (dimensions == 1) {
      for (i = 0; i < Array.getLength(array); i++) {
	if (i > 0)
	  result.append(",");
	if (Array.get(array, i) == null) {
	  result.append("null");
	}
	else {
	  obj = Array.get(array, i);
	  if (outputClass) {
	    if (obj instanceof Class)
	      result.append(((Class) obj).getName());
	    else
	      result.append(obj.getClass().getName());
	  }
	  else {
	    result.append(obj.toString());
	  }
	}
      }
    }
    else {
      for (i = 0; i < Array.getLength(array); i++) {
	if (i > 0)
	  result.append(",");
	result.append("[" + arrayToString(Array.get(array, i)) + "]");
      }
    }

    return result.toString();
  }

  /**
   * Returns the given Array in a string representation. Even though the
   * parameter is of type "Object" one can hand over primitve arrays, e.g.
   * int[3] or double[2][4].
   *
   * @param array       the array to return in a string representation
   * @return            the array as string
   */
  public static String arrayToString(Object array) {
    return arrayToString(array, false);
  }

  /**
   * Converts specified characters into the string equivalents.
   *
   * @param string 	the string
   * @param find	the characters to replace
   * @param replace	the replacement strings for the characters
   * @return 		the converted string
   * @see		#unbackQuoteChars(String, String[], char[])
   */
  public static String backQuoteChars(String string, char[] find, String[] replace) {
    int 		index;
    StringBuilder 	newStr;
    int			i;

    for (i = 0; i < find.length; i++) {
      if (string.indexOf(find[i]) != -1 ) {
	newStr = new StringBuilder();
	while ((index = string.indexOf(find[i])) != -1) {
	  if (index > 0)
	    newStr.append(string.substring(0, index));
	  newStr.append(replace[i]);
	  if ((index + 1) < string.length())
	    string = string.substring(index + 1);
	  else
	    string = "";
	}
	newStr.append(string);
	string = newStr.toString();
      }
    }

    return string;
  }

  /**
   * Converts carriage returns and new lines in a string into \r and \n.
   * Backquotes the following characters: ` " \ \t and %
   *
   * @param string 	the string
   * @return 		the converted string
   * @see		#unbackQuoteChars(String)
   */
  public static String backQuoteChars(String string) {
    return backQuoteChars(
	string,
	new char[]  {'\\',   '\'',  '\t',  '\n',  '\r',  '"'},
	new String[]{"\\\\", "\\'", "\\t", "\\n", "\\r", "\\\""});
  }

  /**
   * The inverse operation of backQuoteChars().
   * Converts the specified strings into their character representations.
   *
   * @param string 	the string
   * @param find	the string to find
   * @param replace	the character equivalents of the strings
   * @return 		the converted string
   * @see		#backQuoteChars(String, char[], String[])
   */
  public static String unbackQuoteChars(String string, String[] find, char[] replace) {
    int 		index;
    StringBuilder 	newStr;
    int[] 		pos;
    int			curPos;
    String 		str;
    int			i;

    pos = new int[find.length];

    str = new String(string);
    newStr = new StringBuilder();
    while (str.length() > 0) {
      // get positions and closest character to replace
      curPos = str.length();
      index  = -1;
      for (i = 0; i < pos.length; i++) {
	pos[i] = str.indexOf(find[i]);
	if ( (pos[i] > -1) && (pos[i] < curPos) ) {
	  index  = i;
	  curPos = pos[i];
	}
      }

      // replace character if found, otherwise finished
      if (index == -1) {
	newStr.append(str);
	str = "";
      }
      else {
	newStr.append(str.substring(0, pos[index]));
	newStr.append(replace[index]);
	str = str.substring(pos[index] + find[index].length());
      }
    }

    return newStr.toString();
  }

  /**
   * The inverse operation of backQuoteChars().
   * Converts back-quoted carriage returns and new lines in a string
   * to the corresponding character ('\r' and '\n').
   * Also "un"-back-quotes the following characters: ` " \ \t and %
   *
   * @param string 	the string
   * @return 		the converted string
   * @see		#backQuoteChars(String)
   */
  public static String unbackQuoteChars(String string) {
    return unbackQuoteChars(
	string,
	new String[]{"\\\\", "\\'", "\\t", "\\n", "\\r", "\\\""},
	new char[]  {'\\',   '\'',  '\t',  '\n',  '\r',  '"'});
  }

  /**
   * Quotes a string if it contains special characters.
   *
   * The following rules are applied:
   *
   * A character is backquoted version of it is one
   * of <tt>" ' % \ \n \r \t</tt>.
   *
   * A string is enclosed within double quotes if a character has been
   * backquoted using the previous rule above or contains
   * <tt>{ }</tt> or is exactly equal to the strings
   * <tt>, ? space or ""</tt> (empty string).
   *
   * A quoted question mark distinguishes it from the missing value which
   * is represented as an unquoted question mark in arff files.
   *
   * @param string 	the string to be quoted
   * @return 		the string (possibly quoted)
   * @see		#unDoubleQuote(String)
   */
  public static String doubleQuote(String string) {
    return quote(string, "\"");
  }

  /**
   * Quotes a string if it contains special characters.
   *
   * The following rules are applied:
   *
   * A character is backquoted version of it is one
   * of <tt>" ' % \ \n \r \t</tt>.
   *
   * A string is enclosed within single quotes if a character has been
   * backquoted using the previous rule above or contains
   * <tt>{ }</tt> or is exactly equal to the strings
   * <tt>, ? space or ""</tt> (empty string).
   *
   * A quoted question mark distinguishes it from the missing value which
   * is represented as an unquoted question mark in arff files.
   *
   * @param string 	the string to be quoted
   * @return 		the string (possibly quoted)
   * @see		#unquote(String)
   */
  public static String quote(String string) {
    return quote(string, "'");
  }

  /**
   * Quotes a string if it contains special characters.
   *
   * The following rules are applied:
   *
   * A character is backquoted version of it is one
   * of <tt>" ' % \ \n \r \t</tt>.
   *
   * A string is enclosed within the quote character if a character has been
   * backquoted using the previous rule above or contains
   * <tt>{ }</tt> or is exactly equal to the strings
   * <tt>, ? space or ""</tt> (empty string).
   *
   * A quoted question mark distinguishes it from the missing value which
   * is represented as an unquoted question mark in arff files.
   *
   * @param string 	the string to be quoted
   * @param quoteChar	the quote character to use
   * @return 		the string (possibly quoted)
   * @see		#unquote(String,String)
   */
  public static String quote(String string, String quoteChar) {
      boolean quote = false;

      // backquote the following characters
      if ((string.indexOf('\n') != -1) || (string.indexOf('\r') != -1) ||
	  (string.indexOf('\'') != -1) || (string.indexOf('"')  != -1) ||
	  (string.indexOf('\\') != -1) || (string.indexOf('\t') != -1)) {
	  string = backQuoteChars(string);
	  quote = true;
      }

      // Enclose the string in quotes if the string contains a recently added
      // backquote or contains one of the following characters.
      if((quote == true) ||
	 (string.indexOf('{') != -1) || (string.indexOf('}') != -1) ||
	 (string.indexOf(',') != -1) || (string.equals("?")) ||
	 (string.indexOf(' ') != -1) || (string.equals(""))) {
	  string = (new String(quoteChar).concat(string)).concat(new String(quoteChar));
      }

      return string;
  }

  /**
   * unquotes are previously quoted string (but only if necessary), i.e., it
   * removes the double quotes around it. Inverse to doubleQuote(String).
   *
   * @param string	the string to process
   * @return		the unquoted string
   * @see		#doubleQuote(String)
   */
  public static String unDoubleQuote(String string) {
    return unquote(string, "\"");
  }

  /**
   * unquotes are previously quoted string (but only if necessary), i.e., it
   * removes the single quotes around it. Inverse to quote(String).
   *
   * @param string	the string to process
   * @return		the unquoted string
   * @see		#quote(String)
   */
  public static String unquote(String string) {
    return unquote(string, "'");
  }

  /**
   * unquotes are previously quoted string (but only if necessary), i.e., it
   * removes the quote characters around it. Inverse to quote(String,String).
   *
   * @param string	the string to process
   * @param quoteChar	the quote character to use
   * @return		the unquoted string
   * @see		#quote(String,String)
   */
  public static String unquote(String string, String quoteChar) {
    if (string.startsWith(quoteChar) && string.endsWith(quoteChar)) {
      string = string.substring(1, string.length() - 1);

      if ((string.indexOf("\\n")  != -1) || (string.indexOf("\\r")  != -1) ||
	  (string.indexOf("\\'")  != -1) || (string.indexOf("\\\"") != -1) ||
	  (string.indexOf("\\\\") != -1) || (string.indexOf("\\t")  != -1)) {
	string = unbackQuoteChars(string);
      }
    }

    return string;
  }

  /**
   * Creates a deep copy of the given object (must be serializable!). Returns
   * null in case of an error.
   *
   * @param o		the object to copy
   * @return		the deep copy
   */
  public static Object deepCopy(Object o) {
    Object		result;
    SerializedObject	so;

    try {
      so     = new SerializedObject((Serializable) o);
      result = so.getObject();
    }
    catch (Exception e) {
      System.err.println("Failed to serialize " + o.getClass().getName() + ":");
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Creates a new instance of the class represented by this object.
   *
   * @param o		the object to create a new instance for
   * @return		the new instance, or null in case of an error
   */
  public static Object newInstance(Object o) {
    Object	result;

    try {
      result = o.getClass().newInstance();
    }
    catch (Exception e) {
      System.err.println("Error creating new instance for " + o.getClass().getName() + ":");
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Breaks up the string, if wider than "columns" characters.
   *
   * @param s		the string to process
   * @param columns	the width in columns
   * @return		the processed string
   */
  public static String[] breakUp(String s, int columns) {
    Vector<String>	result;
    String		line;
    BreakIterator	boundary;
    int			boundaryStart;
    int			boundaryEnd;
    String		word;
    String		punctuation;
    int			i;
    String[]		lines;

    result      = new Vector<String>();
    punctuation = " .,;:!?'\"";
    lines       = s.split("\n");

    for (i = 0; i < lines.length; i++) {
      boundary      = BreakIterator.getWordInstance();
      boundary.setText(lines[i]);
      boundaryStart = boundary.first();
      boundaryEnd   = boundary.next();
      line          = "";

      while (boundaryEnd != BreakIterator.DONE) {
	word = lines[i].substring(boundaryStart, boundaryEnd);
	if (line.length() >= columns) {
	  if (word.length() == 1) {
	    if (punctuation.indexOf(word.charAt(0)) > -1) {
	      line += word;
	      word = "";
	    }
	  }
	  result.add(line);
	  line = "";
	}
	line          += word;
	boundaryStart  = boundaryEnd;
	boundaryEnd    = boundary.next();
      }
      if (line.length() > 0)
	result.add(line);
    }

    return result.toArray(new String[result.size()]);
  }

  /**
   * Inserts line breaks into the string, if wider than "columns" characters.
   *
   * @param s		the string to process
   * @param columns	the width in columns
   * @return		the processed string
   */
  public static String insertLineBreaks(String s, int columns) {
    StringBuffer	result;
    String[]		lines;
    int			i;

    result = new StringBuffer();

    lines = breakUp(s, columns);
    for (i = 0; i < lines.length; i++) {
      if (i > 0)
	result.append("\n");
      result.append(lines[i]);
    }

    return result.toString();
  }

  /**
   * Shortens a string (and appends "...") if longer than the allowed
   * maximum number of characters.
   *
   * @param s		the string to process
   * @param max		the maximum number of characters.
   * @return		the processed string
   */
  public static String shorten(String s, int max) {
    if (s.length() > max)
      return s.substring(0, max) + "...";
    else
      return s;
  }

  /**
   * Load file and return contents as byte array.
   *
   * @param file	file to load
   * @return		byte array
   */
  public static byte[] loadByteArrayFromFile(File file) {
    byte[] 			result;
    BufferedInputStream		bis;
    ByteArrayOutputStream	bytesIn;
    int 			ch;

    bis     = null;
    bytesIn = null;
    try {
      bis     = new BufferedInputStream(new FileInputStream(file));
      bytesIn = new ByteArrayOutputStream();
      while ((ch = bis.read()) != -1)
	bytesIn.write(ch);
      bis.close();
      result = bytesIn.toByteArray();
    } 
    catch(Exception e) {
      result = null;
      e.printStackTrace();
    }
    
    return result;
  }

  /**
   * Inserts comment characters at the start of each line.
   *
   * @param s		the string to process
   * @param comment	the comment string
   * @return		the processed string
   */
  public static String commentOut(String s, String comment) {
    return indent(s, comment);
  }

  /**
   * Inserts blanks at the start of each line.
   *
   * @param s		the string to process
   * @param numBlanks	the number of blanks to insert
   * @return		the processed string
   */
  public static String indent(String s, int numBlanks) {
    StringBuilder	indent;
    int			i;

    indent = new StringBuilder();
    for (i = 0; i < numBlanks; i++)
      indent.append(" ");

    return indent(s, indent.toString());
  }

  /**
   * Indents each line with the specified string.
   *
   * @param s		the string to process
   * @param indentStr	the string to use for indentation
   * @return		the processed string
   */
  protected static String indent(String s, String indentStr) {
    StringBuffer	result;
    String[]		lines;
    int			i;

    result = new StringBuffer();

    lines = s.split("\n");
    for (i = 0; i < lines.length; i++) {
      result.append(indentStr);
      result.append(lines[i]);
      result.append("\n");
    }

    return result.toString();
  }

  /**
   * Removes the comment characters from the start of each line.
   *
   * @param s		the string to process
   * @param comment	the comment string
   * @return		the processed string
   */
  public static String unComment(String s, String comment) {
    StringBuffer	result;
    String[]		lines;
    int			i;

    result = new StringBuffer();

    lines = s.split("\n");
    for (i = 0; i < lines.length; i++) {
      if (lines[i].startsWith(comment))
	result.append(lines[i].substring(comment.length()));
      else
	result.append(lines[i]);
      result.append("\n");
    }

    return result.toString();
  }

  /**
   * Flattens the vector into a single, long string. The newLine string gets
   * added between the lines, but not after the last one.
   *
   * @param lines	the lines to flatten
   * @param newLine	the separator
   * @return		the generated string
   */
  public static String flatten(List<String> lines, String newLine) {
    return flatten(lines.toArray(new String[lines.size()]), newLine);
  }

  /**
   * Flattens the array into a single, long string. The newLine string gets
   * added between the lines, but not after the last one.
   *
   * @param lines	the lines to flatten
   * @param newLine	the separator
   * @return		the generated string
   */
  public static String flatten(String[] lines, String newLine) {
    StringBuffer	result;
    int			i;

    result = new StringBuffer();

    for (i = 0; i < lines.length; i++) {
      if (i > 0)
	result.append(newLine);
      result.append(lines[i]);
    }

    return result.toString();
  }

  /**
   * Removes empty lines from the vector. Performs trimming before ascertaining
   * whether the line is empty.
   *
   * @param lines	the list to clean up
   */
  public static void removeEmptyLines(Vector<String> lines) {
    removeEmptyLines(lines, true);
  }

  /**
   * Removes empty lines from the vector.
   *
   * @param lines	the list to clean up
   * @param trim	whether to trim the line first before checking whether
   * 			it is empty or not
   */
  public static void removeEmptyLines(Vector<String> lines, boolean trim) {
    int		i;
    String	line;

    i = 0;
    while (i < lines.size()) {
      if (trim)
	line = lines.get(i).trim();
      else
	line = lines.get(i);

      if (line.length() == 0)
	lines.remove(i);
      else
	i++;
    }
  }

  /**
   * Removes comment lines from the vector.
   *
   * @param lines	the list to clean up
   * @param comment	the start of a comment
   */
  public static void removeComments(Vector<String> lines, String comment) {
    int		i;

    i = 0;
    while (i < lines.size()) {
      if (lines.get(i).startsWith(comment))
	lines.remove(i);
      else
	i++;
    }
  }

  /**
   * Compares two comparable objects. Takes care of null objects.
   * Returns -1, 0 or +1, if o1 less than, equal to or greater than o2.
   * Returns 0 if both objects null, -1 if o1 null but not o2 and +1 if o1 not
   * null but o2.
   *
   * @param o1		the first object
   * @param o2		the second object
   * @return		the comparison result
   */
  public static int compare(Comparable o1, Comparable o2) {
    if ((o1 != null) && (o2 != null))
      return o1.compareTo(o2);
    else if ((o1 == null) && (o2 == null))
      return 0;
    else if (o1 == null)
      return -1;
    else
      return +1;
  }

  /**
   * Creates a new array of the specified length and fills it with the values
   * of the old array before returning it.
   *
   * @param array	the array to adjust
   * @param newLen	the new length
   * @param defValue	the default value to fill the new array with
   * @return		the fixed array
   */
  public static Object adjustArray(Object array, int newLen, Object defValue) {
    Object	result;
    int		i;
    boolean	serializable;

    serializable = (defValue instanceof Serializable);
    result       = Array.newInstance(defValue.getClass(), newLen);
    for (i = 0; i < Array.getLength(result); i++) {
      if (serializable)
	Array.set(result, i, deepCopy(defValue));
      else
	Array.set(result, i, defValue);
    }

    if (array != null)
	System.arraycopy(
	    array, 0, result, 0,
	    (Array.getLength(array) < Array.getLength(result)) ? Array.getLength(array) : Array.getLength(result));

    return result;
  }

  /**
   * Turns a class into a string.
   *
   * @param c		the class to turn into a string
   * @return		the string
   */
  public static String classToString(Class c) {
    String	result;

    if (c.isArray())
      result = c.getComponentType().getName() + "[]";
    else
      result = c.getName();

    return result;
  }

  /**
   * Turns the double array into a float array.
   *
   * @param array	the array to convert
   * @return		the converted array
   */
  public static float[] toFloat(double[] array) {
    float[]	result;
    int		i;

    result = new float[array.length];
    for (i = 0; i < array.length; i++)
      result[i] = (float) array[i];

    return result;
  }

  /**
   * Turns the float array into a double array.
   *
   * @param array	the array to convert
   * @return		the converted array
   */
  public static double[] toDouble(float[] array) {
    double[]	result;
    int		i;

    result = new double[array.length];
    for (i = 0; i < array.length; i++)
      result[i] = array[i];

    return result;
  }

  /**
   * Pads the string with a padding character with to at most "width" width.
   * Does not truncate the string.
   *
   * @param s		the string to pad
   * @param padding	the padding character
   * @param width	the maximum width
   * @return		the padded string
   */
  public static String padLeft(String s, char padding, int width) {
    return padLeft(s, padding, width, false);
  }

  /**
   * Pads the string with a padding character with to at most "width" width.
   * Truncating, if string is longer than "width", is optional.
   *
   * @param s		the string to pad
   * @param padding	the padding character
   * @param width	the maximum width
   * @param truncate	if true then the string can be truncated (on the left)
   * 			to fit width
   * @return		the padded string
   */
  public static String padLeft(String s, char padding, int width, boolean truncate) {
    StringBuilder	result;

    result = new StringBuilder(s);

    // pad
    while (result.length() < width)
      result.insert(0, padding);

    // truncate
    if (truncate) {
      if (result.length() > width)
	result.delete(0, result.length() - width);
    }

    return result.toString();
  }

  /**
   * Returns the stacktrace of the throwable as string.
   *
   * @param t		the throwable to get the stacktrace for
   * @return		the stacktrace
   */
  public static String throwableToString(Throwable t) {
    return throwableToString(t, -1);
  }

  /**
   * Returns the stacktrace of the throwable as string.
   *
   * @param t		the throwable to get the stacktrace for
   * @param maxLines	the maximum number of lines to print, <= 0 for all
   * @return		the stacktrace
   */
  public static String throwableToString(Throwable t, int maxLines) {
    StringWriter	writer;
    StringBuilder	result;
    String[]		lines;
    int			i;

    writer = new StringWriter();
    t.printStackTrace(new PrintWriter(writer));

    if (maxLines > 0) {
      result = new StringBuilder();
      lines  = writer.toString().split("\n");
      for (i = 0; i < maxLines; i++) {
	if (i > 0)
	  result.append("\n");
	result.append(lines[i]);
      }
    }
    else {
      result = new StringBuilder(writer.toString());
    }

    return result.toString();
  }

  /**
   * Returns the current stack trace.
   *
   * @param maxDepth	the maximum depth of the stack trace, <= 0 for full trace
   * @return		the stack trace as string (multiple lines)
   */
  public static String getStackTrace(int maxDepth) {
    StringBuilder	result;
    Throwable		th;
    StackTraceElement[]	trace;
    int			i;

    result = new StringBuilder();
    th     = new Throwable();
    th.fillInStackTrace();
    trace  = th.getStackTrace();
    if (maxDepth <= 0)
      maxDepth = trace.length - 1;
    maxDepth++;  // we're starting at 1 not 0
    maxDepth = Math.min(maxDepth, trace.length);
    for (i = 1; i < maxDepth; i++) {
      if (i > 1)
	result.append("\n");
      result.append(trace[i]);
    }

    return result.toString();
  }

  /**
   * Converts the given decimal number into a different base.
   *
   * @param n		the decimal number to convert
   * @param base	the base
   * @return		the digits in the new base; index refers to power,
   * 			ie, 0 = base^0, 3 = base^3
   */
  public static Vector<Integer> toBase(int n, int base) {
    Vector<Integer>	result;
    int			current;
    int			times;
    int			remainder;

    result  = new Vector<Integer>();
    current = n;
    do {
      times     = current / base;
      remainder = current - times * base;
      result.add(remainder);
      current   = times;
    }
    while (times > 0);

    return result;
  }

  /**
   * Returns a hexadecimal representation of the byte value.
   * <p/>
   * Taken from <a href="http://www.herongyang.com/Cryptography/SHA1-Message-Digest-in-Java.html" target="_blank">here</a>.
   *
   * @param value	the value to convert
   * @return		the hexadecimal representation
   */
  @MixedCopyright(
      copyright = "2006 Dr. Herong Yang",
      author = "Dr. Herong Yang",
      license = "provided as-is",
      url = "http://www.herongyang.com/Cryptography/SHA1-Message-Digest-in-Java.html"
  )
  public static String toHex(byte value) {
    StringBuilder	result;

    result = new StringBuilder();
    result.append(HEX_DIGIT[(value >> 4) & 0x0f]);
    result.append(HEX_DIGIT[(value     ) & 0x0f]);

    return result.toString();
  }

  /**
   * Tries to parse the given string as byte.
   *
   * @param s		the string to check
   * @return		true if it represents a valid byte
   */
  public static boolean isByte(String s) {
    try {
      Byte.parseByte(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Tries to parse the given string as short.
   *
   * @param s		the string to check
   * @return		true if it represents a valid short
   */
  public static boolean isShort(String s) {
    try {
      Short.parseShort(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Tries to parse the given string as integer.
   *
   * @param s		the string to check
   * @return		true if it represents a valid integer
   */
  public static boolean isInteger(String s) {
    try {
      Integer.parseInt(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Tries to parse the given string as long.
   *
   * @param s		the string to check
   * @return		true if it represents a valid long
   */
  public static boolean isLong(String s) {
    try {
      Long.parseLong(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Tries to parse the given string as float.
   *
   * @param s		the string to check
   * @return		true if it represents a valid float
   */
  public static boolean isFloat(String s) {
    try {
      Float.parseFloat(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Tries to parse the given string as double.
   *
   * @param s		the string to check
   * @return		true if it represents a valid double
   */
  public static boolean isDouble(String s) {
    try {
      Double.parseDouble(s);
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Splits the row into separate cells based on the delimiter character.
   * String.split(regexp) does not work with empty cells (eg ",,,").
   *
   * @param line	the row to split
   * @param delimiter	the delimiter to use
   * @return		the cells
   */
  public static String[] split(String line, char delimiter) {
    return split(line, "" + delimiter);
  }

  /**
   * Splits the row into separate cells based on the delimiter string.
   * String.split(regexp) does not work with empty cells (eg ",,,").
   *
   * @param line	the row to split
   * @param delimiter	the delimiter to use
   * @return		the cells
   */
  public static String[] split(String line, String delimiter) {
    ArrayList<String>	result;
    int			lastPos;
    int			currPos;

    result  = new ArrayList<String>();
    lastPos = -1;
    while ((currPos = line.indexOf(delimiter, lastPos + 1)) > -1) {
      result.add(line.substring(lastPos + 1, currPos));
      lastPos = currPos;
    }
    result.add(line.substring(lastPos + 1));

    return result.toArray(new String[result.size()]);
  }
}
