/**
 * Log.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.Serializable;

import adams.core.Properties;
import adams.env.Environment;
import adams.env.LogDefinition;

/**
 * Ancestor class for simple logging support.

 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2815 $
 */
public abstract class Log
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = -3404920421654324738L;

  /** the name of the props file. */
  public final static String FILENAME = "Log.props";

  /** the key for the default class. */
  public final static String KEY_DEFAULTCLASS = "DefaultClass";

  /** the key for the enabled/disabled. */
  public final static String KEY_ENABLED = "Enabled";

  /** the underlying properties. */
  protected static Properties m_Properties;

  /** the default logging object. */
  protected static Log m_Singleton;

  /** whether logging is enabled or not. */
  protected boolean m_Enabled;

  /**
   * Initializes the logger.
   */
  public Log() {
    super();
    initialize();
  }

  /**
   * Initializes the logger.
   */
  protected void initialize() {
    m_Enabled = getProperties().getBoolean(KEY_ENABLED, false);
  }

  /**
   * Logs the message.
   *
   * @param msg		the message to log
   * @param out		whether to log default output (= true) or error output (= false)
   * @param newline	whether to add a newline or not
   */
  protected abstract void doLog(String msg, boolean out, boolean newline);

  /**
   * Logs the string to the default output (no newline).
   *
   * @param msg		the message to log
   */
  public void printOut(String msg) {
    if (m_Enabled)
      doLog(msg, true, false);
  }

  /**
   * Logs the string to the error output (no newline).
   *
   * @param msg		the message to log
   */
  public void printErr(String msg) {
    if (m_Enabled)
      doLog(msg, false, false);
  }

  /**
   * Logs the string to the default output (with newline).
   *
   * @param msg		the message to log
   */
  public void printlnOut(String msg) {
    if (m_Enabled)
      doLog(msg, true, true);
  }

  /**
   * Logs the string to the error output (with newline).
   *
   * @param msg		the message to log
   */
  public void printlnErr(String msg) {
    if (m_Enabled)
      doLog(msg, false, true);
  }

  /**
   * Returns the underlying properties for logging.
   *
   * @return		the properties
   */
  protected static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(LogDefinition.KEY);

    return m_Properties;
  }

  /**
   * Returns a log class specific property key.
   *
   * @param suffix	the key to retrieve (classname plus a dot get prefixed
   * 			to this suffix)
   * @return		the property key
   */
  protected String getPropertyKey(String suffix) {
    return getClass().getName() + "." + suffix;
  }

  /**
   * Returns the singleton to use for logging.
   *
   * @return		the singleton
   */
  public static synchronized Log getSingleton() {
    String	classname;

    if (m_Singleton == null) {
      try {
	classname   = getProperties().getString(KEY_DEFAULTCLASS);
	m_Singleton = (Log) Class.forName(classname).newInstance();
      }
      catch (Exception e) {
	e.printStackTrace();
	m_Singleton = new ConsoleLog();
      }
    }

    return m_Singleton;
  }
}
