/*
 * Fit.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.data.fit;

import adams.core.ClassLister;
import adams.core.ConsoleObject;
import adams.core.option.OptionHandler;
import adams.core.option.OptionManager;
import adams.core.option.OptionUtils;

/**
 * Abstract super class of methods that fit a function to 2-D data.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3136 $
 */
public abstract class Fit
  extends ConsoleObject
  implements OptionHandler {

  /** for serialization. */
  private static final long serialVersionUID = 1392766177324738497L;

  /** below this threshold, coefficients are thought to be zero.
   * @see #fitClean(double[], double[], double[], double[], double[]) */
  public final static double SMALL = 10E-8;

  /** for managing the available options. */
  protected OptionManager m_OptionManager;

  /** whether debugging is on. */
  protected boolean m_Debug;

  /**
   * Instantiates the fitting method with the given basis function.
   */
  public Fit() {
    super();
    initialize();
    defineOptions();
    getOptionManager().setDefaults();
  }

  /**
   * Returns a short description of the fitting method.
   *
   * @return		a description of the method
   */
  public abstract String globalInfo();

  /**
   * Initializes the members.
   */
  protected void initialize() {
  }

  /**
   * Returns a new instance of the option manager.
   *
   * @return		the manager to use
   */
  protected OptionManager newOptionManager() {
    return new OptionManager(this);
  }

  /**
   * Adds options to the internal list of options. Derived classes must
   * override this method to add additional options.
   */
  public void defineOptions() {
    m_OptionManager = newOptionManager();

    m_OptionManager.add(
	    "D", "debug", false);
  }

  /**
   * Returns the option manager.
   *
   * @return		the manager
   */
  public OptionManager getOptionManager() {
    if (m_OptionManager == null)
      defineOptions();

    return m_OptionManager;
  }

  /**
   * Cleans up the options.
   */
  public void cleanUpOptions() {
    if (m_OptionManager != null) {
      m_OptionManager.cleanUp();
      m_OptionManager = null;
    }
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Cleans up the options.
   *
   * @see	#cleanUpOptions()
   */
  public void destroy() {
    cleanUpOptions();
  }

  /**
   * Set debugging mode.
   *
   * @param value 	true if debug output should be printed
   */
  public void setDebug(boolean value) {
    m_Debug = value;
    getDebugging().setEnabled(value);
  }

  /**
   * Returns whether debugging is turned on.
   *
   * @return 		true if debugging output is on
   */
  public boolean getDebug() {
    return m_Debug;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String debugTipText() {
    return "If set to true, scheme may output additional info to the console.";
  }

  /**
   * Returns a description of the fitting method.
   *
   * @return		a description or name
   */
  public abstract String getDescription();

  /**
   * Fits a function to 2-D data for measurements
   * y[i] vs. dependent variable x[i].
   *
   * @param parameters 	first half of the array holds the coefficients for the function.
   * 			The second half holds the errors on the coefficients.
   * @param x		independent variable
   * @param y 		vertical dependent variable
   * @param sigma_x 	std. dev. error on each x value; use null to ignore array
   * @param sigma_y 	std. dev. error on each y value; use null to ignore array
   * @return		true if successfully computed
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public abstract boolean fit(double[] parameters, double[] x, double[] y, double[] sigma_x, double[] sigma_y);

  /**
   * Fits a function to 2-D data for measurements
   * y[i] vs. dependent variable x[i].
   *
   * In addition to the the <code>fit(...)</code> method, this method sets
   * coefficients explicitly to zero if their absolute value is smaller than
   * <code>SMALL</code>.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @param y 		vertical dependent variable
   * @param sigma_x 	std. dev. error on each x value; use null to ignore array
   * @param sigma_y 	std. dev. error on each y value; use null to ignore array
   * @return		true if successfully computed
   * @see		#fit(double[], double[], double[], double[], double[])
   * @see		#SMALL
   */
  public boolean fitClean(double[] parameters, double[] x, double[] y, double[] sigma_x, double[] sigma_y) {
    boolean	result;
    int		i;

    result = fit(parameters, x, y, sigma_x, sigma_y);
    for (i = 0; i< parameters.length; i++) {
      if (Math.abs(parameters[i]) < SMALL)
	parameters[i] = 0.0;
    }

    return result;
  }

  /**
   * Calculates a single y value based on the provided parameters and x value.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @return		the calculated y
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public abstract double calculate(double[] parameters, double x);

  /**
   * Calculates the y values based on the provided parameters and x values.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @return		the calculated y's
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public double[] calculate(double[] parameters, double[] x) {
    double[]	result;
    int		i;

    result = new double[x.length];

    for (i = 0; i < x.length; i++)
      result[i] = calculate(parameters, x[i]);

    return result;
  }

  /**
   * Returns whether the initial parameters can be guessed.
   *
   * @return		true if the initial parameters can be guessed
   */
  public abstract boolean canGuess();

  /**
   * Returns the guessed initial parameters, if that is supported.
   *
   * @param x		the x values
   * @param y		the y values
   * @return		the initial parameters, null if not supported
   */
  public abstract double[] guess(double[] x, double[] y);

  /**
   * Returns a short description.
   *
   * @return		short description
   */
  public abstract String toString();

  /**
   * Returns a list with classnames of fitting algorithms.
   *
   * @return		the fitting algorithm classnames
   */
  public static String[] getFits() {
    return ClassLister.getSingleton().getClassnames(Fit.class);
  }

  /**
   * Instantiates the fitting algorithm with the given options.
   *
   * @param classname	the classname of the fitting algorithm to instantiate
   * @param options	the options for the fitting algorithm
   * @return		the instantiated fitting algorithm or null if an error occurred
   */
  public static Fit forName(String classname, String[] options) {
    Fit		result;

    try {
      result = (Fit) OptionUtils.forName(Fit.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }
}