/*
 * NonlinearLeastSquares.java
 * Copyright (C) 1992 William H. Press, Saul A. Teukolsky, William T. Vetterling, Brian P. Flannery (= Numerical Recipes in C)
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand - port to Java
 */

package adams.data.fit;

import adams.core.ConsoleObject;
import adams.core.TechnicalInformation;
import adams.core.TechnicalInformationHandler;
import adams.core.TechnicalInformation.Field;
import adams.core.TechnicalInformation.Type;
import adams.core.annotation.MixedCopyright;
import adams.core.option.OptionUtils;

/**
 <!-- globalinfo-start -->
 * Performs Levenberg-Marquardt non-linear least squares curve fitting.<br/>
 * <br/>
 * For more information, see:<br/>
 * <br/>
 * William H. Press, Saul A. Teukolsky, William T. Vetterling, Brian P. Flannery (1992). Nonlinear Models.<br/>
 * <br/>
 * WikiPedia. Non-linear least squares. URL http://en.wikipedia.org/wiki/Non-linear_least_squares.<br/>
 * <br/>
 * WikiPedia. Singular value decomposition. URL http://en.wikipedia.org/wiki/Singular_value_decomposition.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- technical-bibtex-start -->
 * BibTeX:
 * <pre>
 * &#64;inbook{Press1992,
 *    author = {William H. Press and Saul A. Teukolsky and William T. Vetterling and Brian P. Flannery},
 *    chapter = {15.5},
 *    edition = {Second},
 *    pages = {681-688},
 *    publisher = {Cambridge University Press},
 *    series = {Numerical Recipes in C},
 *    title = {Nonlinear Models},
 *    year = {1992},
 *    PDF = {http://www.nrbook.com/a/bookcpdf/c15-4.pdf}
 * }
 *
 * &#64;misc{missing_id,
 *    author = {WikiPedia},
 *    title = {Non-linear least squares},
 *    URL = {http://en.wikipedia.org/wiki/Non-linear_least_squares}
 * }
 *
 * &#64;misc{missing_id,
 *    author = {WikiPedia},
 *    title = {Singular value decomposition},
 *    URL = {http://en.wikipedia.org/wiki/Singular_value_decomposition}
 * }
 * </pre>
 * <p/>
 <!-- technical-bibtex-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-function &lt;adams.core.fit.NonlinearFunction [options]&gt; (property: function)
 *         The function to use for calculating the y values based on the x input.
 *         default: adams.core.fit.Exp2
 * </pre>
 *
 * Default options for adams.core.fit.Exp2 (-function/function):
 *
 * <pre>-D (property: debug)
 *         If set to true, function may output additional info to the console.
 * </pre>
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4436 $
 */
@MixedCopyright(
    copyright = "1992 William H. Press, Saul A. Teukolsky, William T. Vetterling, Brian P. Flannery (= Numerical Recipes in C)"
)
public class NonlinearLeastSquares
  extends Fit
  implements TechnicalInformationHandler {

  /**
   * The "Levenberg-Marquardt" least squares method.
   *
   <!-- technical-bibtex-start -->
   * BibTeX:
   * <pre>
   * &#64;inbook{Press1992,
   *    author = {William H. Press and Saul A. Teukolsky and William T. Vetterling and Brian P. Flannery},
   *    chapter = {15.5},
   *    edition = {Second},
   *    pages = {681-688},
   *    publisher = {Cambridge University Press},
   *    series = {Numerical Recipes in C},
   *    title = {Nonlinear Models},
   *    year = {1992},
   *    PDF = {http://www.nrbook.com/a/bookcpdf/c15-4.pdf}
   * }
   *
   * &#64;misc{missing_id,
   *    author = {WikiPedia},
   *    title = {Non-linear least squares},
   *    URL = {http://en.wikipedia.org/wiki/Non-linear_least_squares}
   * }
   *
   * &#64;misc{missing_id,
   *    author = {WikiPedia},
   *    title = {Singular value decomposition},
   *    URL = {http://en.wikipedia.org/wiki/Singular_value_decomposition}
   * }
   * </pre>
   * <p/>
   <!-- technical-bibtex-end -->
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4436 $
   */
  public static class LevenbergMarquardt
    extends ConsoleObject
    implements TechnicalInformationHandler {

    /** for serialization. */
    private static final long serialVersionUID = 1153571818457692622L;

    /** counts how many cofficients need to be adapted. */
    protected int mfit;

    /** helper variable. */
    protected double[] da;

    /** the old chisq values. */
    protected double[] ochisq;

    /** test values for the coefficients. */
    protected double[] atry;

    /** the beta variable. */
    protected double[] beta;

    /** for gauss-jordan, the right-hand side vector. */
    protected double[][] oneda;

    /**
     * Returns an instance of a TechnicalInformation object, containing
     * detailed information about the technical background of this class,
     * e.g., paper reference or book this class is based on.
     *
     * @return 		the technical information about this class
     */
    public TechnicalInformation getTechnicalInformation() {
      TechnicalInformation 	result;
      TechnicalInformation 	additional;

      result = new TechnicalInformation(Type.INBOOK);
      result.setValue(Field.AUTHOR, "William H. Press and Saul A. Teukolsky and William T. Vetterling and Brian P. Flannery");
      result.setValue(Field.SERIES, "Numerical Recipes in C");
      result.setValue(Field.EDITION, "Second");
      result.setValue(Field.TITLE, "Nonlinear Models");
      result.setValue(Field.CHAPTER, "15.5");
      result.setValue(Field.PAGES, "681-688");
      result.setValue(Field.YEAR, "1992");
      result.setValue(Field.PUBLISHER, "Cambridge University Press");
      result.setValue(Field.PDF, "http://www.nrbook.com/a/bookcpdf/c15-4.pdf");

      additional = result.add(Type.MISC);
      additional.setValue(Field.AUTHOR, "WikiPedia");
      additional.setValue(Field.TITLE, "Non-linear least squares");
      additional.setValue(Field.URL, "http://en.wikipedia.org/wiki/Non-linear_least_squares");

      additional = result.add(Type.MISC);
      additional.setValue(Field.AUTHOR, "WikiPedia");
      additional.setValue(Field.TITLE, "Singular value decomposition");
      additional.setValue(Field.URL, "http://en.wikipedia.org/wiki/Singular_value_decomposition");

      return result;
    }

    /**
     * Linear equation solution by Gauss-Jordan elimination, equation (2.1.1)
     * above. a[0..n-1][0..n-1] is the input matrix. b[0..n-1][0..m-1] is input containing
     * the m right-hand side vectors. On output, a is replaced by its matrix
     * inverse, and b is replaced by the corresponding set of solution vectors.
     *
     * @param a		the input matrix
     * @param b		the input containing the m right-hand side vectors
     * @param n		the dimension of a
     * @param m		the dimension of b
     */
    public void gaussj(double a[][], int n, double b[][], int m) {
      int[] 	indxc;
      int[] 	indxr;
      int[]	ipiv;
      int 	i;
      int		icol;
      int		irow;
      int		j;
      int		k;
      int		l;
      int		ll;
      double 	big;
      double	dum;
      double	pivinv;
      double	tmp;

      icol  = 0;
      irow  = 0;
      indxc = new int[n]; // The integer arrays ipiv, indxr,and indxc are
      indxr = new int[n]; // used for bookkeeping on the pivoting.
      ipiv  = new int[n];
      for (j = 0; j < n; j++)
	ipiv[j] = 0;
      for (i = 0; i < n; i++) { // This is the main loop over the columns to be
	big = 0.0; // reduced.
	for (j = 0; j < n; j++) // This is the outer loop of the search for a pivot
	  if (ipiv[j] != 1) // element.
	    for (k = 0; k < n; k++) {
	      if (ipiv[k] == 0) {
		if (Math.abs(a[j][k]) >= big) {
		  big  = Math.abs(a[j][k]);
		  irow = j;
		  icol = k;
		}
	      }
	    }
	++(ipiv[icol]);

	// We now have the pivot element, so we interchange rows, if needed, to put the
	// pivot element on the diagonal. The columns are not physically interchanged,
	// only relabeled: indxc[i],the column of the ith pivot element, is the ith
	// column that is reduced, while indxr[i] is the row in which that pivot
	// element was originally located. If indxr[i] =/=indxc[i] there is an implied
	// column interchange. With this form of bookkeeping, the solution b's will end
	// up in the correct order, and the inverse matrix will be scrambled by
	// columns.

	if (irow != icol) {
	  for (l = 0; l < n; l++) {
	    tmp        = a[irow][l];
	    a[irow][l] = a[icol][l];
	    a[icol][l] = tmp;
	  }
	  for (l = 0; l < m; l++) {
	    tmp        = b[irow][l];
	    b[irow][l] = b[icol][l];
	    b[icol][l] = tmp;
	  };
	}
	indxr[i] = irow; // We are now ready to divide the pivot row by the
	indxc[i] = icol; // pivot element, located at irow and icol.
	if (a[icol][icol] == 0.0) {
	  getSystemErr().println("gaussj: Singular Matrix");
	  return;
	}
	pivinv        = 1.0 / a[icol][icol];
	a[icol][icol] = 1.0;
	for (l = 0; l < n; l++)
	  a[icol][l] *= pivinv;
	for (l = 0; l < m; l++)
	  b[icol][l] *= pivinv;

	for (ll = 0; ll < n; ll++) // Next, we reduce the rows...
	  if (ll != icol) { // ...except for the pivot one, of course.
	    dum         = a[ll][icol];
	    a[ll][icol] = 0.0;
	    for (l = 0; l < n; l++)
	      a[ll][l] -= a[icol][l]*dum;
	    for (l = 0; l < m; l++)
	      b[ll][l] -= b[icol][l]*dum;
	  }
      }

      // This is the end of the main loop over columns of the reduction. It only
      // remains to unscramble the solution in view of the column interchanges. We do
      // this by interchanging pairs of columns in the reverse order that the
      // permutation was built up.

      for (l = n - 1;l >= 0;l--) {
	if (indxr[l] != indxc[l])
	  for (k = 0; k < n; k++) {
	    tmp            = a[k][indxr[l]];
	    a[k][indxr[l]] = a[k][indxc[l]];
	    a[k][indxc[l]] = tmp;
	  }
      } // And we are done.
    }

    /**
     * Expand in storage the covariance matrix covar, so as to take into account
     * parameters that are being held fixed. (For the latter, return zero
     * covariances.)
     *
     * @param covar	the covariance matrix
     * @param ma	the number of coefficients
     * @param ia	a "0" indicates to leave the coefficient alone,
     * 			a "1" means the coefficient can be adapted,
     * 			dimension [0..ma-1]
     * @param mfit	the number of coefficients to fit
     */
    public void covsrt(double[][] covar, int ma, int ia[], int mfit) {
      int 	i;
      int		j;
      int		k;
      double 	swap;

      for (i = mfit; i < ma; i++) {
	for (j = 0; j < i; j++) {
	  covar[i][j] = 0.0;
	  covar[j][i] = 0.0;
	}
      }

      k = mfit - 1;

      for (j = ma - 1; j >= 0; j--) {
	if (ia[j] == 1) {
	  for (i = 0; i < ma; i++) {
	    swap        = covar[i][k];
	    covar[i][k] = covar[i][j];
	    covar[i][j] = swap;
	  }
	  for (i = 0; i < ma; i++) {
	    swap        = covar[k][i];
	    covar[k][i] = covar[j][i];
	    covar[j][i] = swap;
	  }
	  k--;
	}
      }
    }

    /**
     * Used by mrqmin to evaluate the linearized fitting matrix alpha, and vector
     * beta as in (15.5.8), and calculate chi^2 .
     *
     * @param x		the x values, dimension [0..ndata-1]
     * @param y		the y values to fit, dimension [0..ndata-1]
     * @param sig	the sigma values for y, dimension [0..ndata-1]
     * @param ndata	the number of data points
     * @param a		the array for the determined coefficients,
     * 			dimension [0..ma-1]
     * @param ia	a "0" indicates that this parameter should left alone,
     * 			a "1" that one can adjust it,
     * 			dimension [0..ma-1]
     * @param ma	the number of coefficients
     * @param alpha 	the alpha matrix
     * @param beta	the beta matrix
     * @param chisq	the chi^2 value
     * @param function	the function to calculate the y from x and the partial derivatives
     */
    public void mrqcof(double x[], double y[], double sig[], int ndata, double a[], int ia[], int ma, double[][] alpha, double beta[], double[] chisq, NonlinearFunction function) {
      int 	i;
      int	j;
      int	k;
      int	l;
      int	m;
      int	mfit;
      double 	ymod;
      double	wt;
      double	sig2i;
      double	dy;
      double[] 	dyda;

      mfit = 0;
      dyda = new double[ma];
      for (j = 0; j < ma; j++) {
	if (ia[j] == 1)
	  mfit++;
      }
      for (j = 0; j < mfit; j++) { // Initialize (symmetric) alpha, beta.
	for (k = 0; k <= j; k++)
	  alpha[j][k] = 0.0;
	beta[j] = 0.0;
      }
      chisq[0] = 0.0;
      for (i = 0; i < ndata; i++) { // Summation loop over all data.
	dyda = function.calcDerivatives(x[i], a);
	ymod = function.calcY(x[i], a);
	if ((sig != null) && (sig[i] != 0.0))
	  sig2i = 1.0 / (sig[i]*sig[i]);
	else
	  sig2i = 1.0;
	dy    = y[i] - ymod;
	for (j = -1, l = 0; l < ma; l++) {
	  if (ia[l] == 1) {
	    wt = dyda[l]*sig2i;
	    for (j++, k = -1, m = 0; m <= l; m++) {
	      if (ia[m] == 1)
		alpha[j][++k] += wt*dyda[m];
	    }
	    beta[j] += dy*wt;
	  }
	}
	chisq[0] += dy*dy*sig2i; // And find chi^2 .
      }
      for (j = 1;j < mfit; j++) // Fill in the symmetric side.
	for (k = 0; k < j; k++)
	  alpha[k][j] = alpha[j][k];
    }

    /**
     * Levenberg-Marquardt method, attempting to reduce the value chi^2 of a fit
     * between a set of data points x[0..ndata-1], y[0..ndata-1] with individual
     * standard deviations sig[0..ndata-1], and a nonlinear function dependent on ma
     * coefficients a[0..ma-1]. The input array ia[0..ma-1] indicates by nonzero
     * entries those components of a that should be fitted for, and by zero entries
     * those components that should be held fixed at their input values. The
     * program returns current best-fit values for the parameters a[0..ma-1], and chi^2 =
     * chisq[0]. The arrays covar[0..ma-1][0..ma-1], alpha[0..ma-1][0..ma-1] are used as
     * working space during most iterations. Supply a routine
     * funcs(x,a,yfit,dyda,ma) that evaluates the fitting function yfit, and its
     * derivatives dyda[0..ma-1] with respect to the fitting parameters a at x.On the
     * first call provide an initial guess for the parameters a,and set alamda[0]&lt;0
     * for initialization (which then sets alamda[0]=.001). If a step succeeds chisq[0]
     * becomes smaller and alamda[0] decreases by a factor of 10. If a step fails
     * alamda[0] grows by a factor of 10. You must call this routine repeatedly until
     * convergence is achieved. Then, make one final call with alamda[0]=0,so that
     * covar[0..ma-1][0..ma-1] returns the covariance matrix, and alpha the curvature
     * matrix.  (Parameters held fixed will return zero covariances.)
     *
     * @param x		the x values, dimension: [0..ndata-1]
     * @param y		the y values to fit, dimension: [0..ndata-1]
     * @param sig	the sigma values for y, dimension: [0..ndata-1]
     * @param ndata	the number of data points
     * @param a		the array for the determined coefficients,
     * 			dimension: [0..ma-1]
     * @param ia	a "0" indicates that this parameter should left alone,
     * 			a "1" that one can adjust it,
     * 			dimension: [0..ma-1]
     * @param ma	the number of coefficients
     * @param covar	the covariance matrix, dimension: [0..ma-1][0..ma-1]
     * @param alpha 	the alpha matrix, dimension: [0..ma-1][0..ma-1]
     * @param chisq	the chi^2 value, array of size 1
     * @param function	the function to calculate the y from x and the partial derivatives
     * @param alamda	the alamda parameter, array of size 1
     */
    public void mrqmin(double x[], double y[], double sig[], int ndata, double a[], int ia[], int ma, double[][] covar, double[][] alpha, double[] chisq, NonlinearFunction function, double[] alamda) {
      int 	j;
      int		k;
      int		l;

      if (alamda[0] < 0.0) { // Initialization.
	atry = new double[ma];
	beta = new double[ma];
	da   = new double[ma];
	for (mfit = 0, j = 0;j < ma; j++) {
	  if (ia[j] == 1)
	    mfit++;
	}
	oneda     = new double[mfit][1];
	alamda[0] = 0.001;
	mrqcof(x, y, sig, ndata, a, ia, ma, alpha, beta, chisq, function);
	ochisq = new double[1];
	ochisq[0] = chisq[0];
	for (j = 0; j < ma; j++)
	  atry[j] = a[j];
      }

      for (j = 0; j < mfit; j++) { // Alter linearized fitting matrix, by augmenting di
	for (k = 0; k < mfit; k++)
	  covar[j][k] = alpha[j][k]; // agonal elements.
	covar[j][j] = alpha[j][j] * (1.0 + (alamda[0]));
	oneda[j][0] = beta[j];
      }

      gaussj(covar,mfit,oneda,1); // Matrixsolution.

      for (j = 0; j < mfit; j++)
	da[j] = oneda[j][0];

      if (alamda[0] == 0.0) { // Once converged, evaluate covariance matrix.
	covsrt(covar, ma, ia, mfit);
	covsrt(alpha, ma, ia, mfit); // Spread out alpha to its full size too.
	return;
      }

      for (j = -1, l = 0; l < ma; l++) { // Did the trial succeed?
	if (ia[l] == 1)
	  atry[l] = a[l] + da[++j];
      }
      mrqcof(x, y, sig, ndata, atry, ia, ma, covar, da, chisq, function);
      if (chisq[0] < ochisq[0]) { // Success, accept the new solution.
	alamda[0] *= 0.1;
	ochisq[0] = chisq[0];
	for (j = 0; j < mfit; j++) {
	  for (k = 0; k < mfit; k++)
	    alpha[j][k] = covar[j][k];
	  beta[j] = da[j];
	}
	for (l = 0; l < ma; l++)
	  a[l] = atry[l];
      }
      else { // Failure, increase alamda and return.
	alamda[0] *= 10.0;
	chisq[0]   = ochisq[0];
      }
    }
  }

  /** for serialization. */
  private static final long serialVersionUID = 8031789267977329707L;

  /** the function to use for calculating, i.e., f(x) = y. */
  protected NonlinearFunction m_Function;

  /**
   * Returns a short description of the fitting method.
   *
   * @return		a description of the method
   */
  public String globalInfo() {
    return
        "Performs Levenberg-Marquardt non-linear least squares curve fitting.\n\n"
      + "For more information, see:\n\n"
      + getTechnicalInformation().toString();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"function", "function",
	new Exp2());
  }

  /**
   * Returns a description of the fitting method.
   *
   * @return		a description or name
   */
  public String getDescription() {
    return "Non-linear least squares";
  }

  /**
   * Returns an instance of a TechnicalInformation object, containing
   * detailed information about the technical background of this class,
   * e.g., paper reference or book this class is based on.
   *
   * @return 		the technical information about this class
   */
  public TechnicalInformation getTechnicalInformation() {
    return new LevenbergMarquardt().getTechnicalInformation();
  }

  /**
   * Sets the function for calculating the y values based on x, i.e., f(x) = ...
   *
   * @param value	the function to use
   */
  public void setFunction(NonlinearFunction value) {
    if (value != null)
      m_Function = value;
    else
      getSystemErr().println("Function cannot be null!");
  }

  /**
   * Returns the function for calculating the y values based on x, i.e., f(x) = ...
   *
   * @return		the function
   */
  public NonlinearFunction getFunction() {
    return m_Function;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String functionTipText() {
    return "The function to use for calculating the y values based on the x input.";
  }

  /**
   * Fits a function to 2-D data for measurements
   * y[i] vs. dependent variable x[i].
   *
   * @param parameters 	holds the coefficients.
   * 			Needs to contain initial guesses of parameters.
   * @param x		independent variable
   * @param y 		vertical dependent variable
   * @param sigma_x 	std. dev. error on each x value; use null to ignore array
   * @param sigma_y 	std. dev. error on each y value; use null to ignore array
   * @return		true if successfully converged
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public boolean fit(double[] parameters, double[] x, double[] y, double[] sigma_x, double[] sigma_y) {
    boolean		result;
    LevenbergMarquardt	lm;
    double[]		alamda;
    double[]		chisq;
    double[]		a;
    int[] 		ia;
    int 		ma;
    double[][] 		alpha;
    double[][] 		covar;
    int			i;

    // setup parameters
    lm     = new LevenbergMarquardt();
    alamda = new double[1];
    chisq  = new double[1];
    ma     = parameters.length;
    ia     = new int[ma];
    a      = new double[ma];
    alpha  = new double[ma][ma];
    covar  = new double[ma][ma];
    for (i = 0; i < ma; i++) {
      a[i]  = parameters[i];
      ia[i] = 1;
    }

    // try to converge
    alamda[0] = -1.0;
    while (alamda[0] < 1.0)
      lm.mrqmin(x, y, sigma_y, x.length, a, ia, ma, covar, alpha, chisq, m_Function, alamda);
    result = (alamda[0] < 1.0);

    // calculate covar/alpha matrices
    alamda[0] = 0.0;
    lm.mrqmin(x, y, sigma_x, x.length, a, ia, ma, covar, alpha, chisq, m_Function, alamda);

    // save found parameters
    for (i = 0; i < ma; i++)
      parameters[i] = a[i];

    return result;
  }

  /**
   * Calculates the y value based on the provided parameters and x value.
   *
   * @param parameters 	holds the coefficients for the function.
   * @param x		independent variable
   * @return		the calculated y
   * @see		#fitClean(double[], double[], double[], double[], double[])
   */
  public double calculate(double[] parameters, double x) {
    return m_Function.calcY(x, parameters);
  }

  /**
   * Returns whether the initial parameters can be guessed.
   *
   * @return		true if the initial parameters can be guessed
   */
  public boolean canGuess() {
    return (m_Function instanceof InitialParameterGuesser);
  }

  /**
   * Returns the guessed initial parameters, if that is supported.
   *
   * @param x		the x values
   * @param y		the y values
   * @return		the initial parameters, null if not supported
   */
  public double[] guess(double[] x, double[] y) {
    if (canGuess())
      return ((InitialParameterGuesser) m_Function).guess(x, y);
    else
      return null;
  }

  /**
   * Returns a short description.
   *
   * @return		short description
   */
  public String toString() {
    return getDescription() + ": " + OptionUtils.getCommandLine(m_Function);
  }
}
