/**
 * AbstractTable.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.db;

import adams.core.ClassLister;
import adams.event.DatabaseConnectionChangeEvent;
import adams.event.DatabaseConnectionChangeListener;

/**
 * Ancestor for all table classes.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3611 $
 */
public abstract class AbstractTable
  extends SQL
  implements DatabaseConnectionChangeListener {

  /** for serialization. */
  private static final long serialVersionUID = 4511302757992864994L;

  /** name of the table. */
  protected String m_TableName;

  /**
   * Initializes the table.
   *
   * @param dbcon	the database context this table is used in
   * @param tableName	the name of the table
   */
  public AbstractTable(AbstractDatabaseConnection dbcon, String tableName) {
    super(dbcon);

    m_TableName = tableName;
    m_DatabaseConnection.addChangeListener(this);
  }

  /**
   * Get name of table.
   *
   * @return	table name
   */
  public String getTableName() {
    return m_TableName;
  }

  /**
   * Checks that a given table exists.
   *
   * @return true if the table exists.
   */
  protected boolean tableExists() {
    return tableExists(getTableName());
  }

  /**
   * Initialize table, e.g., creating.
   *
   * @return 	successful initialization
   */
  public abstract boolean init();

  /**
   * A change in the database connection occurred. Derived classes can
   * override this method to react to changes in the connection.
   *
   * @param e		the event
   */
  public void databaseConnectionStateChanged(DatabaseConnectionChangeEvent e) {
  }

  /**
   * Returns a short string representation.
   *
   * @return		the string representation
   */
  public String toString() {
    return getTableName() + ": " + getDatabaseConnection().toString();
  }

  /**
   * Returns a list with classnames of tables.
   *
   * @return		the table classnames
   */
  public static String[] getTables() {
    return ClassLister.getSingleton().getClassnames(AbstractTable.class);
  }
}
