/*
 * BulkInsertTable.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.db;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Table that can do bulk inserts.
 *
 * @author dale
 * @version $Revision: 3484 $
 */
public abstract class BulkInsertTable
  extends AbstractIndexedTable{

  /** for serialization. */
  private static final long serialVersionUID = 401732448232750879L;

  /** columns for bulk insert. */
  protected Vector<String> m_cols=new Vector<String>();

  /** debug. **/
  private boolean debug=false;

  /**
   * Constructor. Setup column vector.
   *
   * @param dbcon	the database context this table is used in
   * @param tableName	the name of the table
   */
  public BulkInsertTable(AbstractDatabaseConnection dbcon, String tablename) {
    super(dbcon, tablename);

    ColumnMapping cm=getColumnMapping();
    for (Enumeration enum1 = cm.keys() ; enum1.hasMoreElements() ;) {
      String cname=(String)enum1.nextElement();
      m_cols.add(cname);
    }
  }

  /**
   * Return Insert header. Generated from column names
   *
   * @return insert header
   */
  public String generateInsertHeader() {
    return("INSERT INTO "+getTableName()+" ("+getInsertColumnsAsString()+") VALUES ");
  }

  /**
   * Attach a multiple insert object to this table.
   *
   * @param mi
   */
  public void attach(MultipleInsert mi) {
    mi.setTable(this);
    mi.setColumnVector(this.m_cols);
  }

  /**
   * Insert bulk data.
   *
   * @param mi		multiple insert
   * @param vals	hashtable of values
   * @return	success?
   */
  public boolean insert(MultipleInsert mi, Hashtable<String,String> vals) {
    String insString=mi.insert(vals);
    if (insString!=null) {
      try {
	if (debug) {
	  getSystemErr().println("Entered ms point execute from insert");
	}
	boolean ret=execute(insString);
	if (debug) {
	  getSystemErr().println("Complete ms point execute from insert");
	}
	return(ret);
      } catch (Exception e) {
	// TODO Auto-generated catch block
	getSystemErr().printStackTrace(e);
	return(false);
      }
    }
    return(true);//ok
  }

  /**
   * Complete bulk insert.
   *
   * @param mi
   * @return	success?
   */
  protected boolean doInsert(MultipleInsert mi) {
    String ins=mi.getInsertString();
    if (ins!=null) {
      try {
	if (debug) {
	  getSystemErr().println("Entered ms point execute");
	}
	return(this.execute(ins));
      } catch (Exception e) {
	// TODO Auto-generated catch block
	getSystemErr().printStackTrace(e);
	return(false);
      }
    }
    return(true);
  }

  /**
   * Complete bulk insert.
   *
   * @param mi
   * @return	success?
   */
  public boolean insertComplete(MultipleInsert mi) {
    return(doInsert(mi));
  }

  /**
   * Return columns as comma separated list.
   *
   * @return	column names
   */
  protected String getInsertColumnsAsString() {
    String q="";
    for (int i=0;i<m_cols.size();i++) {
      String val=m_cols.elementAt(i);

      q+=val;
      if (i != m_cols.size()-1) {
	q+=",";
      }
    }
    return(q);
  }
}
