/*
 * BaseTable.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 * Copyright (C) 2004 Addision-Wesley, The Java Developers Almanac 1.4
 * Copyright (C) 2007 Michael Dunn, JavaRanch
 */

package adams.gui.core;

import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashSet;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.SizeSequence;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import adams.core.annotation.MixedCopyright;
import adams.gui.event.RemoveItemsEvent;
import adams.gui.event.RemoveItemsListener;

/**
 * A specialized JTable that allows double-clicking on header for resizing to
 * optimal width.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4436 $
 */
public class BaseTable
  extends JTable {

  /** for serialization. */
  private static final long serialVersionUID = -2360462659067336490L;

  /** for setting the optimal column width. */
  protected JTableHelper m_TableHelper;

  /** whether to use dynamic row height or not. */
  protected boolean m_UseDynamicRowHeight = false;

  /** the model for the row heights. */
  protected SizeSequence m_RowModel;

  /** the listeners for items to be removed. */
  protected HashSet<RemoveItemsListener> m_RemoveItemsListeners;

  /**
   * Constructs a default <code>BaseTable</code> that is initialized with a default
   * data model, a default column model, and a default selection
   * model.
   */
  public BaseTable() {
    super();
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> with <code>numRows</code>
   * and <code>numColumns</code> of empty cells using
   * <code>DefaultTableModel</code>.  The columns will have
   * names of the form "A", "B", "C", etc.
   *
   * @param numRows           the number of rows the table holds
   * @param numColumns        the number of columns the table holds
   */
  public BaseTable(int numRows, int numColumns) {
    super(numRows, numColumns);
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> to display the values in the two dimensional array,
   * <code>rowData</code>, with column names, <code>columnNames</code>.
   * <code>rowData</code> is an array of rows, so the value of the cell at row 1,
   * column 5 can be obtained with the following code:
   * <p>
   * <pre> rowData[1][5]; </pre>
   * <p>
   * All rows must be of the same length as <code>columnNames</code>.
   * <p>
   * @param rowData           the data for the new table
   * @param columnNames       names of each column
   */
  public BaseTable(final Object[][] rowData, final Object[] columnNames) {
    super(rowData, columnNames);
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> to display the values in the
   * <code>Vector</code> of <code>Vectors</code>, <code>rowData</code>,
   * with column names, <code>columnNames</code>.  The
   * <code>Vectors</code> contained in <code>rowData</code>
   * should contain the values for that row. In other words,
   * the value of the cell at row 1, column 5 can be obtained
   * with the following code:
   * <p>
   * <pre>((Vector)rowData.elementAt(1)).elementAt(5);</pre>
   * <p>
   * @param rowData           the data for the new table
   * @param columnNames       names of each column
   */
  public BaseTable(Vector rowData, Vector columnNames) {
    super(rowData, columnNames);
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> that is initialized with
   * <code>dm</code> as the data model, a default column model,
   * and a default selection model.
   *
   * @param dm        the data model for the table
   */
  public BaseTable(TableModel dm) {
    super(dm);
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> that is initialized with
   * <code>dm</code> as the data model, <code>cm</code>
   * as the column model, and a default selection model.
   *
   * @param dm        the data model for the table
   * @param cm        the column model for the table
   */
  public BaseTable(TableModel dm, TableColumnModel cm) {
    super(dm, cm);
    initGUI();
  }

  /**
   * Constructs a <code>BaseTable</code> that is initialized with
   * <code>dm</code> as the data model, <code>cm</code> as the
   * column model, and <code>sm</code> as the selection model.
   * If any of the parameters are <code>null</code> this method
   * will initialize the table with the corresponding default model.
   * The <code>autoCreateColumnsFromModel</code> flag is set to false
   * if <code>cm</code> is non-null, otherwise it is set to true
   * and the column model is populated with suitable
   * <code>TableColumns</code> for the columns in <code>dm</code>.
   *
   * @param dm        the data model for the table
   * @param cm        the column model for the table
   * @param sm        the row selection model for the table
   */
  public BaseTable(TableModel dm, TableColumnModel cm, ListSelectionModel sm) {
    super(dm, cm, sm);
    initGUI();
  }

  /**
   * Returns the table helper instance. Instantiates it if necessary.
   *
   * @return		the table helper
   */
  protected JTableHelper getTableHelper() {
    if (m_TableHelper == null)
      m_TableHelper = new JTableHelper(this);

    return m_TableHelper;
  }

  /**
   * Initializes some GUI-related things.
   */
  protected void initGUI() {
    getTableHeader().addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent e) {
	// just the selecte column
	if (MouseUtils.isDoubleClick(e) && e.isControlDown() && !e.isAltDown() && !e.isShiftDown()) {
	  final int col = columnAtPoint(e.getPoint());
	  if ((col != -1) && isVisible()) {
	    SwingUtilities.invokeLater(new Runnable() {
	      public void run() {
		getTableHelper().setOptimalColumnWidth(col);
	      }
	    });
	  }
	}
	// all columns
	else if (MouseUtils.isDoubleClick(e) && e.isControlDown() && !e.isAltDown() && e.isShiftDown()) {
	  if (isVisible()) {
	    SwingUtilities.invokeLater(new Runnable() {
	      public void run() {
		getTableHelper().setOptimalColumnWidth();
	      }
	    });
	  }
	}
	else {
	  super.mouseClicked(e);
	}
      }
    });

    m_RemoveItemsListeners = new HashSet<RemoveItemsListener>();

    addKeyListener(new KeyListener() {
      public void keyTyped(KeyEvent e) {
	// ignored
      }
      public void keyReleased(KeyEvent e) {
	// ignored
      }
      public void keyPressed(KeyEvent e) {
	if (m_RemoveItemsListeners.size() > 0) {
	  if ((e.getKeyCode() == KeyEvent.VK_DELETE) && (e.getModifiers() == 0)) {
	    e.consume();
	    notifyRemoveItemsListeners(getSelectedRows());
	  }
	}
      }
    });
  }

  /**
   * Sets the optimal column width for all columns. AutoResize must be set
   * to BaseTable.AUTO_RESIZE_OFF.
   */
  public void setOptimalColumnWidth() {
    if (isVisible()) {
      SwingUtilities.invokeLater(new Runnable() {
	public void run() {
	  getTableHelper().setOptimalColumnWidth();
	}
      });
    }
  }

  /**
   * Sets the optimal column width for the specified column. AutoResize must be set
   * to BaseTable.AUTO_RESIZE_OFF.
   *
   * @param column	the column to resize
   */
  public void setOptimalColumnWidth(final int column) {
    if (isVisible()) {
      SwingUtilities.invokeLater(new Runnable() {
	public void run() {
	  getTableHelper().setOptimalColumnWidth(column);
	}
      });
    }
  }

  /**
   * Sets whether to use dynamic row height or not.
   *
   * @param value	if true then dynamic row height is used
   */
  public void setUseDynamicRowHeight(boolean value) {
    m_UseDynamicRowHeight = value;
  }

  /**
   * Returns whether dynamic row height is turned on or not.
   *
   * @return		true if dynamic row height is turned on
   */
  public boolean getUseDynamicRowHeight() {
    return m_UseDynamicRowHeight;
  }

  /**
   * Determines the row height.
   *
   * @param row		the row to determine the height for
   * @return		the height
   */
  @MixedCopyright(
      copyright = "2004 Addision-Wesley, The Java Developers Almanac 1.4",
      url = "http://www.exampledepot.com/egs/javax.swing.table/RowHeight.html"
  )
  protected int determineRowHeight(int row) {
    int			result;
    int			i;
    int			currHeight;
    TableCellRenderer 	renderer;
    Component 		comp;
    int			margin;

    if (!m_UseDynamicRowHeight) {
      result = super.getRowHeight(row);
    }
    else {
      // Get the current default height for this row
      result = super.getRowHeight();

      // Determine tallest cell in the row
      margin = getRowMargin();
      for (i = 0; i < getColumnCount(); i++) {
	renderer   = getCellRenderer(row, i);
	comp       = prepareRenderer(renderer, row, i);
	currHeight = comp.getPreferredSize().height + 2*margin;
	result     = Math.max(result, currHeight);
      }
    }

    return result;
  }

  /**
   * Returns the model for the row heights, if necessary instantiates it first.
   *
   * @return		the model
   */
  protected synchronized SizeSequence getRowModel() {
    int[]	sizes;
    int		i;

    if (m_RowModel == null) {
      if (!m_UseDynamicRowHeight) {
	m_RowModel = new BaseSizeSequence(getRowCount(), getRowHeight());
      }
      else {
	sizes = new int[getRowCount()];
	for (i = 0; i < sizes.length; i++)
	  sizes[i] = determineRowHeight(i);
	m_RowModel = new BaseSizeSequence(sizes);
      }
    }

    return m_RowModel;
  }

  /**
   * Sets the height, in pixels, of all cells to <code>rowHeight</code>,
   * revalidates, and repaints.
   * The height of the cells will be equal to the row height minus
   * the row margin.
   *
   * @param   rowHeight                       new row height
   */
  public void setRowHeight(int rowHeight) {
    m_RowModel = null;
    super.setRowHeight(rowHeight);
  }

  /**
   * Returns the height for the specified row.
   * Original example code taken from <a href="http://www.exampledepot.com/egs/javax.swing.table/RowHeight.html"
   * target="_blank">ExampleDepot/The Java Developers Almanac 1.4</a>.
   *
   * @param row		the row to get the height in pixels for
   * @return		the row height
   */
  @MixedCopyright(
      copyright = "2004 Addision-Wesley, The Java Developers Almanac 1.4",
      url = "http://www.exampledepot.com/egs/javax.swing.table/RowHeight.html"
  )
  public synchronized int getRowHeight(int row) {
    int		result;

    result = getRowModel().getSize(row);
    if (result == -1)
      result = determineRowHeight(row);

    return result;
  }

  /**
   * Returns the index of the row that <code>point</code> lies in,
   * or -1 if the result is not in the range
   * [0, <code>getRowCount()</code>-1].
   *
   * @param   point   the location of interest
   * @return  the index of the row that <code>point</code> lies in,
   *          or -1 if the result is not in the range
   *          [0, <code>getRowCount()</code>-1]
   * @see     #getRowModel()
   */
  public int rowAtPoint(Point point) {
    int		result;

    if (!m_UseDynamicRowHeight) {
      result = super.rowAtPoint(point);
    }
    else {
      result = getRowModel().getIndex(point.y);
      if (result < 0)
	result = -1;
      else if (result >= getRowCount())
	result = -1;
      else
	return result;
    }

    return result;
  }

  /**
   * Scrolls the row into view.
   * <p/>
   * Code taken from <a href="http://www.coderanch.com/t/345263/Swing-AWT-SWT-JFace/java/make-JTable-scroll-specified-column" target="_blank">here</a>.
   *
   * @param row		the row to scroll into view
   */
  @MixedCopyright(
      copyright = "2007 Michael Dunn, JavaRanch",
      url = "http://www.coderanch.com/t/345263/Swing-AWT-SWT-JFace/java/make-JTable-scroll-specified-column"
  )
  public void scrollRowToVisible(int row) {
    scrollRectToVisible(getCellRect(row, 0, true));
  }

  /**
   * Scrolls the column into view.
   * <p/>
   * Code taken from <a href="http://www.coderanch.com/t/345263/Swing-AWT-SWT-JFace/java/make-JTable-scroll-specified-column" target="_blank">here</a>.
   *
   * @param col		the column to scroll into view
   */
  @MixedCopyright(
      copyright = "2007 Michael Dunn, JavaRanch",
      url = "http://www.coderanch.com/t/345263/Swing-AWT-SWT-JFace/java/make-JTable-scroll-specified-column"
  )
  public void scrollColumnToVisible(int col) {
    scrollRectToVisible(getCellRect(0, col, true));
  }

  /**
   * Sets the <code>RowSorter</code>.  <code>RowSorter</code> is used
   * to provide sorting and filtering to a <code>JTable</code>.
   * <p>
   * This method clears the selection and resets any variable row heights.
   * <p>
   * If the underlying model of the <code>RowSorter</code> differs from
   * that of this <code>JTable</code> undefined behavior will result.
   *
   * @param sorter the <code>RowSorter</code>; <code>null</code> turns
   *        sorting off
   */
  public void setRowSorter(RowSorter<? extends TableModel> sorter) {
    m_RowModel = null;
    super.setRowSorter(sorter);
  }

  /**
   * Invoked when this table's <code>TableModel</code> generates
   * a <code>TableModelEvent</code>.
   * The <code>TableModelEvent</code> should be constructed in the
   * coordinate system of the model; the appropriate mapping to the
   * view coordinate system is performed by this <code>JTable</code>
   * when it receives the event.
   * <p>
   * Application code will not use these methods explicitly, they
   * are used internally by <code>JTable</code>.
   * <p>
   * Note that as of 1.3, this method clears the selection, if any.
   *
   * @param e		the event
   */
  public void tableChanged(TableModelEvent e) {
    m_RowModel = null;
    super.tableChanged(e);
  }

  /**
   * Adds the remove items listener to its internal list.
   *
   * @param l		the listener to add
   */
  public void addRemoveItemsListener(RemoveItemsListener l) {
    m_RemoveItemsListeners.add(l);
  }

  /**
   * Removes the remove items listener from its internal list.
   *
   * @param l		the listener to remove
   */
  public void removeRemoveItemsListener(RemoveItemsListener l) {
    m_RemoveItemsListeners.remove(l);
  }

  /**
   * Notifies the remove items listeners about the indices that are to be
   * removed.
   *
   * @param indices	the indices that should get removed
   */
  protected void notifyRemoveItemsListeners(int[] indices) {
    RemoveItemsEvent	event;

    event = new RemoveItemsEvent(this, indices);
    for (RemoveItemsListener l: m_RemoveItemsListeners)
      l.removeItems(event);
  }

  /**
   * Copies either the selected rows to the clipboard.
   */
  public void copyToClipboard() {
    Action 	copy;
    ActionEvent	event;

    copy  = getActionMap().get("copy");
    event = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "");
    copy.actionPerformed(event);  }
}
