/**
 * Editors.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.beans.PropertyEditorManager;
import java.lang.reflect.Array;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import adams.core.ClassLister;
import adams.core.ClassLocator;
import adams.core.Properties;
import adams.env.Environment;
import adams.env.GOEBlacklistDefinition;
import adams.env.GOEEditorsDefinition;

/**
 * Registers all the editors for the GenericObjectEditor/GenericArrayEditor.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2822 $
 */
public class Editors {

  /** the name of the props file. */
  public final static String FILENAME = "Editors.props";

  /** the name of the props file with the blacklisted classes. */
  public final static String BLACKLIST = "Editors.blacklist";

  /** the blacklisted classes. */
  protected static Properties m_BlacklistedClasses;

  /**
   * Generates the list of editors to register, based on the classes listed in
   * the ClassLister.props file.
   *
   * @return		the generated props file
   */
  protected static Properties getAvailableEditors() {
    Properties			result;
    Properties			classes;
    Properties			packages;
    Vector<String>		enums;
    String			goeEditor;
    String			arrEditor;
    String			enmEditor;
    Enumeration<String>		enm;
    String			classname;
    String[]			classnames;
    String			pkg;
    HashSet<String>		uniquePackages;
    Iterator<String>		iter;
    int				i;

    result  = new Properties();

    classes = ClassLister.getSingleton().getClasses();

    // determine the editors
    goeEditor = GenericObjectEditor.class.getName();
    arrEditor = GenericArrayEditor.class.getName();
    enmEditor = EnumEditor.class.getName();

    // general
    result.setProperty(Object.class.getName() + "[]", arrEditor);

    // superclasses
    enm = (Enumeration<String>) classes.propertyNames();
    while (enm.hasMoreElements()) {
      classname = enm.nextElement();
      result.setProperty(classname, goeEditor);
      result.setProperty(classname + "[]", arrEditor);
    }

    // enums
    // 1. generate package list to search for enums
    uniquePackages = new HashSet<String>();
    packages       = ClassLister.getSingleton().getPackages();
    enm            = (Enumeration<String>) packages.propertyNames();
    while (enm.hasMoreElements()) {
      classname  = enm.nextElement();
      classnames = ClassLister.getSingleton().getPackages(classname);
      for (i = 0; i < classnames.length; i++)
        uniquePackages.add(classnames[i]);
    }

    // 2. search for enums
    iter = uniquePackages.iterator();
    while (iter.hasNext()) {
      pkg   = iter.next();
      enums = (Vector<String>) ClassLocator.findInPackage(Enum.class, pkg);
      enm   = enums.elements();
      while (enm.hasMoreElements()) {
        classname = enm.nextElement();
        result.setProperty(classname, enmEditor);
      }
    }

    return result;
  }

  /**
   * registers all the editors.
   */
  public static void registerEditors() {
    Properties 		props;
    Enumeration 	enm;
    String 		name;
    String 		value;
    Class 		baseCls;
    Class		cls;

    // generate list of editors
    props = getAvailableEditors();

    // register all editors
    enm = props.propertyNames();
    while (enm.hasMoreElements()) {
      name  = enm.nextElement().toString();
      value = props.getProperty(name, "");
      try {
        // array class?
        if (name.endsWith("[]")) {
          baseCls = Class.forName(name.substring(0, name.indexOf("[]")));
          cls = Array.newInstance(baseCls, 1).getClass();
        }
        else {
          cls = Class.forName(name);
        }
        // register
        PropertyEditorManager.registerEditor(cls, Class.forName(value));
      }
      catch (Exception e) {
        System.err.println("Problem registering " + name + "/" + value + ": " + e);
      }
    }

    // register the custom editors
    Editors.registerCustomEditors();

    // register editors for basic Java types
    Editors.registerBasicEditors();
  }

  /**
   * Returns the properties storing the blacklisted classes.
   *
   * @return		the blacklisted class properties
   */
  public static Properties getBlacklistedClasses() {
    return Environment.getInstance().read(GOEBlacklistDefinition.KEY);
  }

  /**
   * For registering editors for basic Java types.
   */
  protected static void registerBasicEditors() {
    try {
      // integer
      PropertyEditorManager.registerEditor(Byte.class, ByteEditor.class);
      PropertyEditorManager.registerEditor(Byte.TYPE, ByteEditor.class);
      PropertyEditorManager.registerEditor(Short.class, ShortEditor.class);
      PropertyEditorManager.registerEditor(Short.TYPE, ShortEditor.class);
      PropertyEditorManager.registerEditor(Integer.class, IntegerEditor.class);
      PropertyEditorManager.registerEditor(Integer.TYPE, IntegerEditor.class);

      // float
      PropertyEditorManager.registerEditor(Long.class, LongEditor.class);
      PropertyEditorManager.registerEditor(Long.TYPE, LongEditor.class);
      PropertyEditorManager.registerEditor(Float.class, FloatEditor.class);
      PropertyEditorManager.registerEditor(Float.TYPE, FloatEditor.class);
      PropertyEditorManager.registerEditor(Double.class, DoubleEditor.class);
      PropertyEditorManager.registerEditor(Double.TYPE, DoubleEditor.class);

      // others
      PropertyEditorManager.registerEditor(Boolean.class, BooleanEditor.class);
      PropertyEditorManager.registerEditor(Boolean.TYPE, BooleanEditor.class);
      PropertyEditorManager.registerEditor(String.class, StringEditor.class);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Registers a custom editor for a certain class. Automatically adds the
   * registering of the array editor.
   *
   * @param cls		the class to register the editor for
   * @param clsEditor	the custom editor
   * @return		true if successfully registered
   */
  public static boolean registerCustomEditor(Class cls, Class clsEditor) {
    boolean	result;

    try {
      PropertyEditorManager.registerEditor(cls, clsEditor);
      PropertyEditorManager.registerEditor(Array.newInstance(cls, 1).getClass(), GenericArrayEditor.class);
      result = true;
    }
    catch (Exception e) {
      result = false;
      System.err.println(
          "Registering editor " + clsEditor.getName()
          + " for class " + cls.getName() + " failed:\n" + e);
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Returns the properties file with the custom editors.
   *
   * @return		the props file
   */
  protected static Properties getCustomEditors() {
    return Environment.getInstance().read(GOEEditorsDefinition.KEY);
  }

  /**
   * Registers all custom editors and adds the hooks as well.
   */
  protected static void registerCustomEditors() {
    Properties 		props;
    Enumeration		enm;
    String		classname;
    String		classnameEditor;
    Class		cls;
    Class		clsEditor;

    props = getCustomEditors();
    enm   = props.propertyNames();
    while (enm.hasMoreElements()) {
      classname = (String) enm.nextElement();
      if (classname.indexOf("#") > -1)
        continue;
      classnameEditor = props.getString(classname);

      // obtain classes
      cls = null;
      try {
        if (classname.endsWith("[]")) {
          cls = Class.forName(classname.substring(0, classname.length() - 2));
          cls = Array.newInstance(cls, 0).getClass();
        }
        else {
          cls = Class.forName(classname);
        }
      }
      catch (Exception e) {
        e.printStackTrace();
        System.err.println("Cannot get class for '" + classname + "' - skipped!");
        continue;
      }

      clsEditor = null;
      try {
        clsEditor = Class.forName(classnameEditor);
      }
      catch (Exception e) {
        e.printStackTrace();
        System.err.println("Cannot get class for editor '" + classnameEditor + "' - skipped!");
        continue;
      }

      // register editors
      registerCustomEditor(cls, clsEditor);
    }
  }
}
