/*
 * Cron.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.standalone;

import java.util.Date;

import org.quartz.CronTrigger;
import org.quartz.Job;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.impl.StdSchedulerFactory;

import adams.flow.control.Sequence;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandler;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.sink.Null;

/**
 <!-- globalinfo-start -->
 * Executes an actor according to a pre-defined schedule.<br/>
 * Note: since the actor merely starts the cron scheduler in the background, the actor finishes the execution pretty much immediately. Therefore, the flow needs to be kept alive in order to let the background jobs getting executed. This can be done with a simple WhileLoop actor using 'true' as expression and a nested Sleep actor.<br/>
 * <br/>
 * For more information on the schedulr format see:<br/>
 * http:&#47;&#47;www.quartz-scheduler.org&#47;docs&#47;tutorials&#47;crontrigger.html
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Cron
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-cron-actor &lt;adams.flow.core.AbstractActor [options]&gt; (property: cronActor)
 * &nbsp;&nbsp;&nbsp;The actor to execute according to the cron schedule.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.control.Sequence -actor adams.flow.sink.Null
 * </pre>
 *
 * <pre>-schedule &lt;java.lang.String&gt; (property: schedule)
 * &nbsp;&nbsp;&nbsp;The schedule for execution the cron actor; format 'SECOND MINUTE HOUR DAYOFMONTH
 * &nbsp;&nbsp;&nbsp;MONTH WEEKDAY [YEAR]'.
 * &nbsp;&nbsp;&nbsp;default: 0 0 1 * * ?
 * </pre>
 *
 <!-- options-end -->
 *
 * For more information on the schedule format, see
 * <a href="http://www.quartz-scheduler.org/docs/tutorials/crontrigger.html" target="_blank">CronTrigger Tutorial</a>.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3251 $
 */
public class Cron
  extends AbstractStandalone
  implements ActorHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4670761846363281951L;

  /**
   * Encapsulates a job to run.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3251 $
   */
  public static class CronJob
    implements Job {

    /**
     * Gets executed when the cron event gets triggered.
     *
     * @param context			the context of the execution
     * @throws JobExecutionException	if job fails
     */
    public void execute(JobExecutionContext context) throws JobExecutionException {
      String	result;
      Cron	owner;

      owner = (Cron) context.getJobDetail().getJobDataMap().get(KEY_OWNER);

      result = owner.getCronActor().execute();
      if (result != null) {
	owner.handleError("execute/cron", owner.getFullName() + ": " + result);
	throw new JobExecutionException(owner.getFullName() + ": " + result);
      }
    }
  }

  /** the key for the owner in the JobExecutionContent. */
  public final static String KEY_OWNER = "owner";

  /** for actor that gets executed. */
  protected AbstractActor m_CronActor;

  /** the cron schedule. */
  protected String m_Schedule;

  /** the scheduler. */
  protected Scheduler m_Scheduler;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Executes an actor according to a pre-defined schedule.\n"
      + "Note: since the actor merely starts the cron scheduler in the background, "
      + "the actor finishes the execution pretty much immediately. Therefore, "
      + "the flow needs to be kept alive in order to let the background jobs "
      + "getting executed. This can be done with a simple WhileLoop actor "
      + "using 'true' as expression and a nested Sleep actor.\n"
      + "\n"
      + "For more information on the schedulr format see:\n"
      + "http://www.quartz-scheduler.org/docs/tutorials/crontrigger.html";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    Sequence seq = new Sequence();
    seq.setActors(new AbstractActor[]{new Null()});

    m_OptionManager.add(
	    "cron-actor", "cronActor",
	    seq);

    m_OptionManager.add(
	    "schedule", "schedule",
	    "0 0 1 * * ?");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("schedule");
    if (variable != null)
      result = variable;
    else
      result = m_Schedule;

    return result;
  }

  /**
   * Checks the cron actor before it is set via the setCronActor method.
   * Returns an error message if the actor is not acceptable, null otherwise.
   * <p/>
   * Default implementation always returns null.
   *
   * @param actor	the actor to check
   * @return		null if accepted, otherwise error message
   */
  protected String checkCronActor(AbstractActor actor) {
    return null;
  }

  /**
   * Updates the parent of all actors in this group.
   */
  protected void updateParent() {
    m_CronActor.setParent(null);
    m_CronActor.setParent(this);
  }

  /**
   * Sets the actor to tee-off to.
   *
   * @param value	the actor
   */
  public void setCronActor(AbstractActor value) {
    String	msg;

    msg = checkCronActor(value);
    if (msg == null) {
      m_CronActor = value;
      reset();
      updateParent();
    }
    else {
      throw new IllegalArgumentException(msg);
    }
  }

  /**
   * Returns the actor to tee-off to.
   *
   * @return		the actor
   */
  public AbstractActor getCronActor() {
    return m_CronActor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String cronActorTipText() {
    return "The actor to execute according to the cron schedule.";
  }

  /**
   * Sets the execution schedule.
   *
   * @param value 	the schedule
   */
  public void setSchedule(String value) {
    m_Schedule = value;
    reset();
  }

  /**
   * Returns the name of the actor.
   *
   * @return 		the name
   */
  public String getSchedule() {
    return m_Schedule;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String scheduleTipText() {
    return
        "The schedule for execution the cron actor; "
      + "format 'SECOND MINUTE HOUR DAYOFMONTH MONTH WEEKDAY [YEAR]'.";
  }

  /**
   * Returns the size of the group.
   *
   * @return		always 1
   */
  public int size() {
    return 1;
  }

  /**
   * Returns the actor at the given position.
   *
   * @param index	the position
   * @return		the actor
   */
  public AbstractActor get(int index) {
    if (index == 0)
      return m_CronActor;
    else
      throw new IndexOutOfBoundsException("Only one item available, requested index: " + index);
  }

  /**
   * Sets the actor at the given position.
   *
   * @param index	the position
   * @param actor	the actor to set at this position
   */
  public void set(int index, AbstractActor actor) {
    if (index == 0)
      setCronActor(actor);
    else
      getSystemErr().println("Index out of range: " + index);
  }

  /**
   * Returns the index of the actor.
   *
   * @param actor	the name of the actor to look for
   * @return		the index of -1 if not found
   */
  public int indexOf(String actor) {
    if (m_CronActor.getName().equals(actor))
      return 0;
    else
      return -1;
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(false, ActorExecution.SEQUENTIAL, false);
  }

  /**
   * Returns the number of non-skipped actors.
   *
   * @return		the 'active' actors
   */
  public int active() {
    if (m_CronActor.getSkip())
      return 0;
    else
      return 1;
  }

  /**
   * Returns the first non-skipped actor.
   *
   * @return		the first 'active' actor, null if none available
   */
  public AbstractActor firstActive() {
    if (m_CronActor.getSkip())
      return null;
    else
      return m_CronActor;
  }

  /**
   * Returns the last non-skipped actor.
   *
   * @return		the last 'active' actor, null if none available
   */
  public AbstractActor lastActive() {
    if (m_CronActor.getSkip())
      return null;
    else
      return m_CronActor;
  }

  /**
   * Performs checks on the "sub-actors". Default implementation does nothing.
   *
   * @return		null
   */
  public String check() {
    return null;
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      result = m_CronActor.setUp();

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    String	msg;
    JobDetail	job;
    CronTrigger	trigger;
    Date	first;

    result = null;

    try {
      m_Scheduler = new StdSchedulerFactory().getScheduler();
      job         = new JobDetail(getFullName() + ".job", getFullName() + ".group", CronJob.class);
      job.getJobDataMap().put(KEY_OWNER, this);
      trigger     = new CronTrigger(
	  getFullName() + ".trigger",
	  getFullName() + ".group",
	  getFullName() + ".job",
	  getFullName() + ".group",
	  m_Schedule);
      m_Scheduler.addJob(job, true);
      first = m_Scheduler.scheduleJob(trigger);
      if (isDebugOn())
	debug("First execution of actor: " + first);
      m_Scheduler.start();
    }
    catch (Exception e) {
      msg = "Failed to set up cron job: ";
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
      result = msg + e;
    }

    return result;
  }

  /**
   * Stops the internal cron scheduler, if possible.
   */
  protected void stopScheduler() {
    if (m_Scheduler != null) {
      try {
	m_Scheduler.shutdown(true);
      }
      catch (Exception e) {
	getSystemErr().println("Error shutting down scheduler:");
	getSystemErr().printStackTrace(e);
      }
    }
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    m_CronActor.stopExecution();
    stopScheduler();
    super.stopExecution();
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    if (m_CronActor != null)
      m_CronActor.wrapUp();
    stopScheduler();
    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    if (m_CronActor != null)
      m_CronActor.cleanUp();

    super.cleanUp();
  }
}
