/**
 * ExcelSpreadSheetWriter.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.OutputStream;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import adams.env.Environment;

/**
 <!-- globalinfo-start -->
 * Writes MS Excel files.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-missing &lt;java.lang.String&gt; (property: missingValue)
 * &nbsp;&nbsp;&nbsp;The placeholder for missing values.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3896 $
 */
public class ExcelSpreadSheetWriter
  extends AbstractSpreadSheetWriter {

  /** for serialization. */
  private static final long serialVersionUID = -3549185519778801930L;

  /** the binary file extension. */
  public static String FILE_EXTENSION = ".xls";

  /** the OOXML file extension. */
  public static String FILE_EXTENSION_OOXML = ".xlsx";

  /** whether to write OOXML instead of binary Excel files (latter is default). */
  protected boolean m_WriteOOXML;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Writes MS Excel files.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "MS Excel spreadsheets";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{"xls", "xlsx"};
  }

  /**
   * Returns whether to write to an OutputStream rather than a Writer.
   *
   * @return		true if to write to an OutputStream
   */
  protected boolean getUseOutputStream() {
    return true;
  }

  /**
   * Sets whether to use OOXML instead of binary Excel files.
   *
   * @param value	if true OOXML files are generated
   */
  public void setWriteOOXML(boolean value) {
    m_WriteOOXML = value;
  }

  /**
   * Returns whether OOXML files are generated instead of binary Excel files.
   *
   * @return		true if OOXML files are generated
   */
  public boolean getWriteOOXML() {
    return m_WriteOOXML;
  }

  /**
   * Writes the spreadsheet in CSV format to the given file.
   *
   * @param content	the spreadsheet to write
   * @param filename	the file to write the spreadsheet to
   * @return		true if successfully written
   */
  public boolean write(SpreadSheet content, String filename) {
    setWriteOOXML(filename.endsWith(FILE_EXTENSION_OOXML));
    return super.write(content, filename);
  }

  /**
   * Performs the actual writing. The caller must ensure that the writer gets
   * closed.
   *
   * @param content	the spreadsheet to write
   * @param out		the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, OutputStream out) {
    boolean		result;
    Workbook 		workbook;
    Sheet 		sheet;
    Row 		row;
    SpreadSheet.Row	spRow;
    SpreadSheet.Cell	spCell;
    Cell 		cell;
    int 		i;
    int 		n;

    result = true;

    try {
      if (getWriteOOXML())
	workbook = new XSSFWorkbook();
      else
	workbook = new HSSFWorkbook();
      sheet = workbook.createSheet();
      workbook.setSheetName(0, Environment.getInstance().getProject());

      // header
      row = sheet.createRow(0);
      for (i = 0; i < content.getColumnCount(); i++) {
	cell = row.createCell(i);
	cell.setCellValue(content.getHeaderRow().getCell(i).getContent());
      }

      // data
      for (n = 0; n < content.getRowCount(); n++) {
	row   = sheet.createRow(n + 1);
	spRow = content.getRow(n);
	for (i = 0; i < content.getColumnCount(); i++) {
	  cell   = row.createCell(i);
	  spCell = spRow.getCell(i);
	  if (spCell.isMissing()) {
	    if (m_MissingValue.length() > 0)
	      cell.setCellValue(m_MissingValue);
	    else
	      cell.setCellType(Cell.CELL_TYPE_BLANK);
	    continue;
	  }

	  if (spCell.isNumeric())
	    cell.setCellValue(Double.parseDouble(spCell.getContent()));
	  else
	    cell.setCellValue(spCell.getContent());
	}
      }

      // save
      workbook.write(out);
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }
}
