/**
 * ExcelSpreadSheetHandler.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.tools.previewbrowser;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.Vector;

import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;

import adams.core.io.ExcelSpreadSheetReader;
import adams.core.io.SpreadSheet;

/**
 <!-- globalinfo-start -->
 * Displays the following spreadsheet types: xls,xlsx
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2752 $
 */
public class ExcelSpreadSheetHandler
  extends AbstractSpreadSheetHandler {

  /** for serialization. */
  private static final long serialVersionUID = -3962259305718630395L;

  /**
   * Returns the list of extensions (without dot) that this handler can
   * take care of.
   *
   * @return		the list of extensions (no dot)
   */
  public String[] getExtensions() {
    return new String[]{"xls", "xlsx"};
  }

  /**
   * Determines the number of sheets in the spreadsheet file.
   *
   * @param file	the spreadsheet file to check
   * @return		the number of sheets
   */
  protected int getSheetCount(File file) {
    int			result;
    Workbook		workbook;
    BufferedInputStream	input;

    input = null;
    try {
      input    = new BufferedInputStream(new FileInputStream(file.getAbsoluteFile()));
      workbook = WorkbookFactory.create(input);
      result   = workbook.getNumberOfSheets();
    }
    catch (Exception e) {
      result = 0;
      getSystemErr().println("Failed to determine sheet count for '" + file + "':");
      getSystemErr().printStackTrace(e);
    }
    finally {
      if (input != null) {
	try {
	  input.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Reads all the invidivual spreadsheets from the file.
   *
   * @param file	the file to read
   * @return		the spreadsheet objects that were read from the file
   */
  protected SpreadSheet[] readAll(File file) {
    Vector<SpreadSheet>		result;
    int				count;
    int				i;
    ExcelSpreadSheetReader	reader;
    SpreadSheet			sheet;

    result = new Vector<SpreadSheet>();
    count  = getSheetCount(file);
    reader = new ExcelSpreadSheetReader();
    for (i = 0; i < count; i++) {
      reader.setSheetIndex("" + (i+1));
      sheet = reader.read(file);
      if (sheet != null)
	result.add(sheet);
    }

    return result.toArray(new SpreadSheet[result.size()]);
  }
}
