/**
 * MultiScriptlet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.gnuplot;

/**
 <!-- globalinfo-start -->
 * Allows the user to chain multiple scriplets together.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-data-file &lt;adams.core.io.PlaceholderFile&gt; (property: dataFile)
 * &nbsp;&nbsp;&nbsp;The data file to use as basis for the plot.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-scriptlet &lt;adams.core.gnuplot.AbstractScriptlet&gt; [-scriptlet ...] (property: scriptlets)
 * &nbsp;&nbsp;&nbsp;The scriplets to use for producing a single script.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-use-single-data-file (property: useSingleDataFile)
 * &nbsp;&nbsp;&nbsp;If enabled, all sub-scriptlets get automatically updated to use this scriptlets
 * &nbsp;&nbsp;&nbsp;data file.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2876 $
 */
public class MultiScriptlet
  extends AbstractScriptlet {

  /** for serialization. */
  private static final long serialVersionUID = 6639840731369734498L;

  /** the custom script code. */
  protected AbstractScriptlet[] m_Scriptlets;

  /** whether to enforce all scriptlets to use the same data file. */
  protected boolean m_UseSingleDataFile;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Allows the user to chain multiple scriplets together.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "scriptlet", "scriptlets",
	    new AbstractScriptlet[0]);

    m_OptionManager.add(
	    "use-single-data-file", "useSingleDataFile",
	    false);
  }

  /**
   * Sets the scriptlets to use.
   *
   * @param value	the scriptlets
   */
  public void setScriptlets(AbstractScriptlet[] value) {
    m_Scriptlets = value;
    reset();
  }

  /**
   * Returns the scriptlets in use.
   *
   * @return		the scriptlets
   */
  public AbstractScriptlet[] getScriptlets() {
    return m_Scriptlets;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String scriptletsTipText() {
    return "The scriplets to use for producing a single script.";
  }

  /**
   * Sets whether to use a single data file only.
   *
   * @param value	if true only a single data file is used
   */
  public void setUseSingleDataFile(boolean value) {
    m_UseSingleDataFile = value;
    reset();
  }

  /**
   * Returns whether to use a single data file only.
   *
   * @return		true if only a single data file is used
   */
  public boolean getUseSingleDataFile() {
    return m_UseSingleDataFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String useSingleDataFileTipText() {
    return "If enabled, all sub-scriptlets get automatically updated to use this scriptlets data file.";
  }

  /**
   * Hook method for performing checks.
   * <p/>
   * Calls the check() methods of all scriptlets.
   *
   * @return		null if all checks passed, otherwise error message
   */
  public String check() {
    String	result;
    int		i;

    result = super.check();

    if (result == null) {
      if (m_UseSingleDataFile) {
	for (i = 0; i < m_Scriptlets.length; i++)
	  m_Scriptlets[i].setDataFile(getDataFile());
      }

      for (i = 0; i < m_Scriptlets.length; i++) {
	result = m_Scriptlets[i].check();
	if (result != null)
	  break;
      }
    }

    return result;
  }

  /**
   * Generates the actual script code.
   *
   * @return		the script code, null in case of an error
   */
  protected String doGenerate() {
    StringBuilder	result;
    int			i;

    result = new StringBuilder();

    for (i = 0; i < m_Scriptlets.length; i++) {
      result.append(COMMENT + " " + m_Scriptlets[i].getClass().getName() + "\n");
      result.append(m_Scriptlets[i].generate());
      result.append("\n");
    }

    return result.toString();
  }
}
