/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    AbstractGroovyActor.java
 *    Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.flow.core;

import adams.core.scripting.Groovy;

/**
 * Abstract ancestor for actors that execute Groovy scripts.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @see Groovy
 */
public abstract class AbstractGroovyActor
  extends AbstractScriptActor {

  /** for serialization. */
  private static final long serialVersionUID = -5904986133981940404L;

  /** the loaded script object. */
  protected transient AbstractActor m_ActorObject;

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptOptionsTipText() {
    return
        "The options for the Groovy script; must consist of 'key=value' pairs "
      + "separated by blanks; the value of 'key' can be accessed via the "
      + "'getAdditionalOptions().getXYZ(\"key\")' method in the Groovy actor.";
  }

  /**
   * Loads the scripts object and sets its options.
   *
   * @return		null if OK, otherwise the error message
   */
  protected String loadScriptObject() {
    String	result;

    result = null;

    try {
      m_ScriptObject = null;
      if (m_ScriptFile.isFile()) {
	m_ScriptObject = (AbstractActor) Groovy.newInstance(m_ScriptFile, AbstractActor.class);
	if (m_ScriptObject != null) {
	  if (!AdditionalOptionsHandlerUtils.setOptions(m_ScriptObject, m_ScriptOptions))
	    result = "Does not implement '" + AdditionalOptionsHandler.class.getName() + "': " + m_ScriptFile;
	}
	else {
	  result = "Failed to instantiate script '" + m_ScriptFile + "'!";
	}
      }
      else {
	result = "No script provided!";
      }
    }
    catch (Exception e) {
      m_ScriptObject = null;
      getSystemErr().printStackTrace(e);
      result = "Failed to initialize Groovy script '" + m_ScriptFile + "': " + e.toString();
    }

    return result;
  }

  /**
   * Checks the script object.
   *
   * @return		null if OK, otherwise the error message
   */
  protected String checkScriptObject() {
    String		result;
    AbstractActor	script;

    result = null;
    script = (AbstractActor) m_ScriptObject;

    if (ActorUtils.isStandalone(this) && !ActorUtils.isStandalone(script))
      result = "Script object is not a singleton!";
    else if (ActorUtils.isSource(this) && !ActorUtils.isSource(script))
      result = "Script object is not a source!";
    else if (ActorUtils.isTransformer(this) && !ActorUtils.isTransformer(script))
      result = "Script object is not a transformer!";
    else if (ActorUtils.isSink(this) && !ActorUtils.isSink(script))
      result = "Script object is not a sink!";

    return result;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      m_ActorObject = (AbstractActor) m_ScriptObject;
      result        = m_ActorObject.setUp();
    }

    return result;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    super.wrapUp();

    if (m_ActorObject != null)
      m_ActorObject.destroy();
    m_ActorObject = null;
  }
}
