/**
 * Image.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.data.imagej;

import ij.ImagePlus;
import adams.core.CloneHandler;
import adams.core.Properties;
import adams.data.Notes;
import adams.data.NotesHandler;

/**
 * A container for ImageJ's ImagePlus objects. It offers additional meta-data.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 1582 $
 */
public class Image
  implements CloneHandler<Image>, NotesHandler {

  /** the key for the filename in the meta-data. */
  public final static String META_FILENAME = "filename";

  /** the key for the class in the meta-data. */
  public final static String META_CLASSVALUE = "class-value";

  /** the key for the class in the meta-data (see constants in weka.core.Attribute class). */
  public final static String META_CLASSTYPE = "class-type";

  /** the stored image. */
  protected ImagePlus m_Image;

  /** the meta-data. */
  protected Properties m_MetaData;

  /** the notes. */
  protected Notes m_Notes;

  /**
   * Initializes the Image container.
   *
   * @param img		the image to encapsulate
   */
  public Image(ImagePlus img) {
    this(img, new Properties());
  }

  /**
   * Initializes the Image container.
   *
   * @param img		the image to encapsulate
   * @param meta	the meta-data to use
   */
  public Image(ImagePlus img, Properties meta) {
    super();

    m_Notes = new Notes();
    setImage(img);
    setMetaData(meta);
  }

  /**
   * Returns a clone of the Image container (image + meta-data are cloned).
   *
   * @return		the clone
   */
  public Image getClone() {
    return new Image(
	new ImagePlus(m_Image.getTitle() + "'", m_Image.getImage()),
	m_MetaData.getClone());
  }

  /**
   * Sets the image to use.
   *
   * @param value	the image
   */
  public void setImage(ImagePlus value) {
    if (value == null)
      throw new IllegalArgumentException("Null image provided!");
    m_Image = value;
  }

  /**
   * Returns the store image.
   *
   * @return		the image
   */
  public ImagePlus getImage() {
    return m_Image;
  }

  /**
   * Sets the meta-data to use (clones it).
   *
   * @param value	the meta-data
   */
  public void setMetaData(Properties value) {
    if (value == null)
      m_MetaData = new Properties();
    else
      m_MetaData = value.getClone();
  }

  /**
   * Returns the current meta-data.
   *
   * @return		the meta-data
   */
  public Properties getMetaData() {
    return m_MetaData;
  }

  /**
   * Returns the currently stored notes.
   *
   * @return		the current notes
   */
  public Notes getNotes() {
    return m_Notes;
  }

  /**
   * Returns a string representation of the container.
   *
   * @return		the string representation
   */
  public String toString() {
    return "img=" + m_Image + ", meta=" + m_MetaData;
  }
}
