/**
 * AbstractJAITransformerTestCase.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.data.jai.transformer;

import java.awt.image.BufferedImage;

import java.io.File;
import java.util.Arrays;
import java.util.Vector;

import javax.media.jai.RenderedOp;

import adams.core.BufferedImageHelper;
import adams.core.CleanUpHandler;
import adams.core.Destroyable;
import adams.core.JAIHelper;
import adams.core.Properties;
import adams.core.Utils;
import adams.core.io.FileUtils;
import adams.data.imagej.Image;
import adams.test.AbstractTestHelper;
import adams.test.AdamsTestCase;
import adams.test.TestHelper;
import adams.test.TmpFile;

/**
 * Ancestor for test cases tailored for JAI transformers.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3702 $
 */
public abstract class AbstractJAITransformerTestCase
  extends AdamsTestCase {

  /**
   * Constructs the test case. Called by subclasses.
   *
   * @param name 	the name of the test
   */
  public AbstractJAITransformerTestCase(String name) {
    super(name);
  }

  /**
   * Returns the test helper class to use.
   *
   * @return		the helper class instance
   */
  protected AbstractTestHelper newTestHelper() {
    return new TestHelper(this, "adams/data/jai/transformer/data");
  }

  /**
   * Loads the image to process.
   *
   * @param filename	the filename to load (without path)
   * @return		the image, null in case of an error
   */
  protected BufferedImage load(String filename) {
    String		fullName;
    RenderedOp		op;
    BufferedImage	result;

    result = null;

    m_TestHelper.copyResourceToTmp(filename);
    fullName = m_TestHelper.getTmpDirectory() + File.separator + filename;
    op       = JAIHelper.read(fullName);
    if (op != null)
      result = op.getAsBufferedImage();
    m_TestHelper.deleteFileFromTmp(filename);

    return result;
  }

  /**
   * Processes the input data and returns the processed data.
   *
   * @param img		the image to work on
   * @param scheme	the scheme to process the data with
   * @return		the generated data
   */
  protected BufferedImage[] process(BufferedImage img, AbstractJAITransformer scheme) {
    return scheme.transform(img);
  }

  /**
   * Saves the data in the tmp directory.
   *
   * @param data	the data to save
   * @param filename	the filename to save to (without path)
   * @return		true if successfully saved
   */
  protected boolean save(BufferedImage data, String filename) {
    TmpFile		file;
    StringBuilder	content;

    file    = new TmpFile(filename);
    content = new StringBuilder();

    content.append("Image:\n");
    content.append(Utils.arrayToString(BufferedImageHelper.getPixelRaster(data)));
    content.append("\n");

    FileUtils.writeToFile(file.getAbsolutePath(), content, false);

    return file.exists();
  }

  /**
   * Returns the filenames (without path) of the input data files to use
   * in the regression test.
   *
   * @return		the filenames
   */
  protected abstract String[] getRegressionInputFiles();

  /**
   * Returns the setups to use in the regression test.
   *
   * @return		the setups
   */
  protected abstract AbstractJAITransformer[] getRegressionSetups();

  /**
   * Creates an output filename based on the input filename.
   *
   * @param input	the input filename (no path)
   * @param no		the number of the test
   * @param imgNo	the number of the image
   * @return		the generated output filename (no path)
   */
  protected String createOutputFilename(String input, int no, int imgNo) {
    String	result;
    int		index;
    String	ext;

    ext = "-out" + no + "_" + imgNo;

    index = input.lastIndexOf('.');
    if (index == -1) {
      result = input + ext;
    }
    else {
      result  = input.substring(0, index);
      result += ext;
    }
    result += ".txt";

    return result;
  }

  /**
   * Compares the processed data against previously saved output data.
   */
  public void testRegression() {
    BufferedImage		data;
    BufferedImage[]		processed;
    boolean			ok;
    String			regression;
    int				i;
    int				n;
    String[]			input;
    AbstractJAITransformer[]	setups;
    String[][]			output;
    Vector<TmpFile>		outputFiles;

    input   = getRegressionInputFiles();
    output  = new String[input.length][];
    setups  = getRegressionSetups();
    assertEquals("Number of files and setups differ!", input.length, setups.length);

    // process data
    for (i = 0; i < input.length; i++) {
      data = load(input[i]);
      assertNotNull("Could not load data for regression test from " + input[i], data);

      processed = process(data, setups[i]);
      assertNotNull("Failed to process data?", processed);

      output[i] = new String[processed.length];
      for (n = 0; n < output[i].length; n++) {
	output[i][n] = createOutputFilename(input[i], i, n);
	ok        = save(processed[n], output[i][n]);
	assertTrue("Failed to save regression data?", ok);
      }
    }

    // test regression
    outputFiles = new Vector<TmpFile>();
    for (i = 0; i < output.length; i++) {
      for (n = 0; n < output[i].length; n++)
	outputFiles.add(new TmpFile(output[i][n]));
    }
    regression = m_Regression.compare(outputFiles.toArray(new TmpFile[outputFiles.size()]));
    assertNull("Output differs:\n" + regression, regression);

    // remove output, clean up scheme
    for (i = 0; i < output.length; i++) {
      if (setups[i] instanceof Destroyable)
	((Destroyable) setups[i]).destroy();
      else if (setups[i] instanceof CleanUpHandler)
	((CleanUpHandler) setups[i]).cleanUp();
      for (n = 0; n < output[i].length; n++)
	m_TestHelper.deleteFileFromTmp(output[i][n]);
    }
  }
}
