/*
 * AbstractImageJFlattener.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.data.imagej.flattener;

import weka.core.Attribute;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.SelectedTag;
import weka.filters.Filter;
import weka.filters.unsupervised.attribute.Add;
import adams.core.ClassLister;
import adams.core.CleanUpHandler;
import adams.core.Properties;
import adams.core.base.BaseString;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;
import adams.data.imagej.Image;

/**
 * Abstract base class for ImageJ flattening operations.
 *
 * Derived classes only have to override the <code>doProcess(ImagePlus)</code>
 * method. The <code>reset()</code> method can be used to reset an
 * algorithms internal state, e.g., after setting options.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 */
public abstract class AbstractImageJFlattener
  extends AbstractOptionHandler
  implements Comparable, CleanUpHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4566948525813804085L;

  /** the prefix for meta-data attributes. */
  public final static String METADATA_PREFIX = "metadata-";

  /** the meta-data fields to convert to numeric attributes. */
  protected BaseString[] m_MetaDataNumeric;

  /** the meta-data fields to convert to string attributes. */
  protected BaseString[] m_MetaDataString;

  /** the current header. */
  protected Instances m_Header;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "meta-numeric", "metaDataNumeric",
	    new BaseString[0]);

    m_OptionManager.add(
	    "meta-string", "metaDataString",
	    new BaseString[0]);
  }

  /**
   * Resets the scheme, i.e., the header information.
   */
  protected void reset() {
    super.reset();

    m_Header = null;
  }

  /**
   * Sets the meta-data properties to convert to numeric attributes.
   *
   * @param value 	the meta-data properties
   */
  public void setMetaDataNumeric(BaseString[] value) {
    m_MetaDataNumeric = value;
    reset();
  }

  /**
   * Returns the meta-data properties to convert to numeric attributes.
   *
   * @return 		the meta-data properties
   */
  public BaseString[] getMetaDataNumeric() {
    return m_MetaDataNumeric;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String metaDataNumericTipText() {
    return "The keys of the meta-data properties that will be converted to numeric attributes.";
  }

  /**
   * Sets the meta-data properties to convert to numeric attributes.
   *
   * @param value 	the meta-data properties
   */
  public void setMetaDataString(BaseString[] value) {
    m_MetaDataString = value;
    reset();
  }

  /**
   * Returns the meta-data properties to convert to numeric attributes.
   *
   * @return 		the meta-data properties
   */
  public BaseString[] getMetaDataString() {
    return m_MetaDataString;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String metaDataStringTipText() {
    return "The keys of the meta-data properties that will be converted to string attributes.";
  }

  /**
   * Optional checks of the image.
   * <p/>
   * Default implementation only checks whether image is null.
   *
   * @param img		the image to check
   */
  protected void checkImage(Image img) {
    if (img == null)
      throw new IllegalStateException("No image provided!");
  }

  /**
   * Creates the header from a template image.
   *
   * @param img		the image to act as a template
   * @return		the generated header
   */
  protected abstract Instances createHeader(Image img);

  /**
   * Adds the meta-data attributes to the header.
   *
   * @param header	the dataset header to work on
   * @param numeric	whether to add numeric or string attributes
   * @return		the new header
   */
  protected Instances createMetaDataHeader(Instances header, boolean numeric) {
    Instances		result;
    BaseString[]	keys;
    String		key;
    int			i;
    Add			add;

    result = header;

    if (numeric)
      keys = m_MetaDataNumeric;
    else
      keys = m_MetaDataString;

    for (i = 0; i < keys.length; i++) {
      key = keys[i].toString();
      add = new Add();
      add.setAttributeIndex("last");
      add.setAttributeName(METADATA_PREFIX + key);
      if (numeric)
	add.setAttributeType(new SelectedTag(Attribute.NUMERIC, Add.TAGS_TYPE));
      else
	add.setAttributeType(new SelectedTag(Attribute.STRING, Add.TAGS_TYPE));
      try {
	add.setInputFormat(result);
	result = Filter.useFilter(result, add);
      }
      catch (Exception e) {
	getSystemErr().println(
	    "Failed to add " + (numeric ? "numeric" : "string" + " attribute for key '" + key + "':"));
	getSystemErr().printStackTrace(e);
      }
    }

    return result;
  }

  /**
   * Adds the meta-data attributes to the header.
   *
   * @param header	the dataset header to work on
   * @return		the new header
   */
  protected Instances createMetaDataHeader(Instances header) {
    Instances	result;

    result = header;
    result = createMetaDataHeader(result, true);
    result = createMetaDataHeader(result, false);

    return result;
  }

  /**
   * Performs the actual flattening of the image. Will use the previously
   * generated header.
   *
   * @param img		the image to process
   * @return		the generated array
   * @see		#m_Header
   */
  protected abstract Instance doFlatten(Image img);

  /**
   * Adds the meta-data values to the instance.
   *
   * @param meta	the meta-data to use
   * @param inst	the instance to process
   * @param numeric	whether numeric or string attributes to add
   * @return		the instance with the meta-data values
   */
  protected Instance addMetaData(Properties meta, Instance inst, boolean numeric) {
    int			i;
    String		key;
    Attribute		att;
    BaseString[]	keys;

    if (numeric)
      keys = m_MetaDataNumeric;
    else
      keys = m_MetaDataString;

    for (i = 0; i < keys.length; i++) {
      key = keys[i].toString();
      // find attribute
      att = m_Header.attribute(METADATA_PREFIX + key);
      if (att == null) {
	debug("Attribute '" + METADATA_PREFIX + key + "' not found??");
	continue;
      }

      // set value
      if (meta.hasKey(key)) {
	if (numeric) {
	  try {
	    inst.setValue(att, meta.getDouble(key));
	  }
	  catch (Exception e) {
	    getSystemErr().println("Failed to parse double value for meta-data key '" + key + "':");
	    getSystemErr().printStackTrace(e);
	  }
	}
	else {
	  inst.setValue(att, meta.getProperty(key));
	}
      }
      // set missing value
      else {
	inst.setMissing(att);
      }
    }

    return inst;
  }

  /**
   * Adds the meta-data values to the instance.
   *
   * @param img		the image to get the meta-data from
   * @param inst	the instance to process
   * @return		the instance with the meta-data values
   */
  protected Instance addMetaData(Image img, Instance inst) {
    Properties	meta;

    meta = img.getMetaData();
    inst = addMetaData(meta, inst, true);
    inst = addMetaData(meta, inst, false);

    return inst;
  }

  /**
   * Process the given image. This method will also create the header if
   * necessary.
   *
   * @param img		the image to process
   * @return		the generated array
   * @see		#m_Header
   * @see		#createHeader(Image)
   */
  public Instance flatten(Image img) {
    Instances	header;
    Instance	result;

    checkImage(img);

    // create header if necessary
    if (m_Header == null) {
      header = createHeader(img);
      if (header == null)
	throw new IllegalStateException("Failed to create header!");
      m_Header = createMetaDataHeader(header);
    }

    result = doFlatten(img);
    result = addMetaData(img, result);

    return result;
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o 	the object to be compared.
   * @return  	a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   *
   * @throws ClassCastException 	if the specified object's type prevents it
   *         				from being compared to this object.
   */
  public int compareTo(Object o) {
    if (o == null)
      return 1;

    return OptionUtils.getCommandLine(this).compareTo(OptionUtils.getCommandLine(o));
  }

  /**
   * Returns whether the two objects are the same.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o	the object to be compared
   * @return	true if the object is the same as this one
   */
  public boolean equals(Object o) {
    return (compareTo(o) == 0);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractImageJFlattener shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractImageJFlattener shallowCopy(boolean expand) {
    return (AbstractImageJFlattener) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    reset();
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Calls cleanUp() and cleans up the options.
   */
  public void destroy() {
    cleanUp();
    super.destroy();
  }

  /**
   * Returns a list with classnames of transformations.
   *
   * @return		the transformation classnames
   */
  public static String[] getFlatteners() {
    return ClassLister.getSingleton().getClassnames(AbstractImageJFlattener.class);
  }

  /**
   * Instantiates the transformation with the given options.
   *
   * @param classname	the classname of the transformation to instantiate
   * @param options	the options for the transformation
   * @return		the instantiated transformation or null if an error occurred
   */
  public static AbstractImageJFlattener forName(String classname, String[] options) {
    AbstractImageJFlattener	result;

    try {
      result = (AbstractImageJFlattener) OptionUtils.forName(AbstractImageJFlattener.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the transformation from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			transformation to instantiate
   * @return		the instantiated transformation
   * 			or null if an error occurred
   */
  public static AbstractImageJFlattener forCommandLine(String cmdline) {
    return (AbstractImageJFlattener) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
