/*
  *    TIFFWriter.java
  *    Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
  *
  */

package adams.gui.print;

import java.io.FileOutputStream;

import javax.media.jai.JAI;

import com.sun.media.jai.codec.TIFFEncodeParam;

/**
 <!-- globalinfo-start -->
 * Outputs TIFF images.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-file &lt;adams.core.io.PlaceholderFile&gt; (property: file)
 * &nbsp;&nbsp;&nbsp;The file to save the image to.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-custom-dimensions (property: useCustomDimensions)
 * &nbsp;&nbsp;&nbsp;Whether to use custom dimensions or use the component's ones.
 * </pre>
 *
 * <pre>-custom-width &lt;int&gt; (property: customWidth)
 * &nbsp;&nbsp;&nbsp;The custom width.
 * &nbsp;&nbsp;&nbsp;default: -1
 * </pre>
 *
 * <pre>-custom-height &lt;int&gt; (property: customHeight)
 * &nbsp;&nbsp;&nbsp;The custom height.
 * &nbsp;&nbsp;&nbsp;default: -1
 * </pre>
 *
 * <pre>-scaling (property: scalingEnabled)
 * &nbsp;&nbsp;&nbsp;If set to true, then scaling will be used.
 * </pre>
 *
 * <pre>-scale-x &lt;double&gt; (property: XScale)
 * &nbsp;&nbsp;&nbsp;The scaling factor for the X-axis.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 * <pre>-scale-y &lt;double&gt; (property: YScale)
 * &nbsp;&nbsp;&nbsp;The scaling factor for the Y axis.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 * <pre>-background &lt;java.awt.Color&gt; (property: background)
 * &nbsp;&nbsp;&nbsp;The background color.
 * &nbsp;&nbsp;&nbsp;default: #ffffff
 * </pre>
 *
 * <pre>-type &lt;RGB|GRAY&gt; (property: type)
 * &nbsp;&nbsp;&nbsp;The type of image to create.
 * &nbsp;&nbsp;&nbsp;default: RGB
 * </pre>
 *
 * <pre>-compress (property: compress)
 * &nbsp;&nbsp;&nbsp;If set to true, the image will be compressed.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3296 $
 */
public class TIFFWriter
  extends BufferedImageBasedWriter {

  /** for serialization. */
  private static final long serialVersionUID = 3324725056211114889L;

  /** whether to compress the image. */
  protected boolean m_Compress;

  /**
   * Returns a string describing the object.
   *
   * @return 		a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Outputs TIFF images.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "compress", "compress",
	    false);
  }

  /**
   * returns the name of the writer, to display in the FileChooser.
   *
   * @return 		the name of the writer
   */
  public String getDescription() {
    return "TIFF-Image";
  }

  /**
   * returns the extensions (incl. ".") of the output format, to use in the
   * FileChooser.
   *
   * @return 		the file extensions
   */
  public String[] getExtensions() {
    return new String[]{".tif", ".tiff"};
  }

  /**
   * Sets whether to compress the image.
   *
   * @param value 	if true then the image will be compressed
   */
  public void setCompress(boolean value) {
    m_Compress = value;
  }

  /**
   * Returns whether to compress the image.
   *
   * @return 		true if the image gets compressed
   */
  public boolean getCompress() {
    return m_Compress;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String compressTipText() {
    return "If set to true, the image will be compressed.";
  }

  /**
   * generates the actual output.
   *
   * @throws Exception	if something goes wrong
   */
  public void generateOutput() throws Exception {
    TIFFEncodeParam 	params;
    FileOutputStream 	os;

    // set up parameters
    params = new com.sun.media.jai.codec.TIFFEncodeParam();
    if (m_Compress)
      params.setCompression(TIFFEncodeParam.COMPRESSION_PACKBITS);

    // save file
    os = new FileOutputStream(getFile().getAbsolutePath());
    JAI.create("encode", createBufferedImage(), os, "TIFF", params);
    os.close();
  }
}
