/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractJAIFlattener.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.data.jai.flattener;

import java.awt.image.BufferedImage;

import weka.core.Instance;
import weka.core.Instances;
import adams.core.ClassLister;
import adams.core.CleanUpHandler;
import adams.core.JAIHelper;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;
import adams.data.imagej.Image;

/**
 * Abstract base class for JAI flattening operations.
 *
 * Derived classes only have to override the <code>doProcess(BufferedImage)</code>
 * method. The <code>reset()</code> method can be used to reset an
 * algorithms internal state, e.g., after setting options.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4649 $
 */
public abstract class AbstractJAIFlattener
  extends AbstractOptionHandler
  implements Comparable, CleanUpHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4566948525813804085L;

  static {
    JAIHelper.disableMediaLib();
  }

  /** the prefix for meta-data attributes. */
  public final static String METADATA_PREFIX = "metadata-";

  /** the current header. */
  protected Instances m_Header;

  /**
   * Resets the scheme, i.e., the header information.
   */
  protected void reset() {
    super.reset();

    m_Header = null;
  }

  /**
   * Optional checks of the image.
   * <p/>
   * Default implementation only checks whether image is null.
   *
   * @param img		the image to check
   */
  protected void checkImage(BufferedImage img) {
    if (img == null)
      throw new IllegalStateException("No image provided!");
  }

  /**
   * Creates the header from a template image.
   *
   * @param img		the image to act as a template
   * @return		the generated header
   */
  protected abstract Instances createHeader(BufferedImage img);

  /**
   * Performs the actual flattening of the image. Will use the previously
   * generated header.
   *
   * @param img		the image to process
   * @return		the generated array
   * @see		#m_Header
   */
  protected abstract Instance doFlatten(BufferedImage img);

  /**
   * Process the given image. This method will also create the header if
   * necessary.
   *
   * @param img		the image to process
   * @return		the generated array
   * @see		#m_Header
   * @see		#createHeader(Image)
   */
  public Instance flatten(BufferedImage img) {
    Instances	header;
    Instance	result;

    checkImage(img);

    // create header if necessary
    if (m_Header == null) {
      header = createHeader(img);
      if (header == null)
	throw new IllegalStateException("Failed to create header!");
      m_Header = header;
    }

    result = doFlatten(img);

    return result;
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o 	the object to be compared.
   * @return  	a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   *
   * @throws ClassCastException 	if the specified object's type prevents it
   *         				from being compared to this object.
   */
  public int compareTo(Object o) {
    if (o == null)
      return 1;

    return OptionUtils.getCommandLine(this).compareTo(OptionUtils.getCommandLine(o));
  }

  /**
   * Returns whether the two objects are the same.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o	the object to be compared
   * @return	true if the object is the same as this one
   */
  public boolean equals(Object o) {
    return (compareTo(o) == 0);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractJAIFlattener shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractJAIFlattener shallowCopy(boolean expand) {
    return (AbstractJAIFlattener) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    reset();
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Calls cleanUp() and cleans up the options.
   */
  public void destroy() {
    cleanUp();
    super.destroy();
  }

  /**
   * Returns a list with classnames of transformations.
   *
   * @return		the transformation classnames
   */
  public static String[] getFlatteners() {
    return ClassLister.getSingleton().getClassnames(AbstractJAIFlattener.class);
  }

  /**
   * Instantiates the transformation with the given options.
   *
   * @param classname	the classname of the transformation to instantiate
   * @param options	the options for the transformation
   * @return		the instantiated transformation or null if an error occurred
   */
  public static AbstractJAIFlattener forName(String classname, String[] options) {
    AbstractJAIFlattener	result;

    try {
      result = (AbstractJAIFlattener) OptionUtils.forName(AbstractJAIFlattener.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the transformation from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			transformation to instantiate
   * @return		the instantiated transformation
   * 			or null if an error occurred
   */
  public static AbstractJAIFlattener forCommandLine(String cmdline) {
    return (AbstractJAIFlattener) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
