/**
 * BufferedImageHelper.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;

/**
 * Helper class for BufferedImage objects.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3702 $
 */
public class BufferedImageHelper {

  /**
   * Returns a copy of a BufferedImage object.
   * <p/>
   * Taken from
   * <a href="http://stackoverflow.com/questions/3514158/how-do-you-clone-a-bufferedimage/3514297#3514297" target="_blank">here</a>
   * (CC BY-SA 3.0).
   *
   * @param img		the image to copy
   */
  public static BufferedImage deepCopy(BufferedImage img) {
    ColorModel 		cm;
    boolean 		isAlphaPremultiplied;
    WritableRaster 	raster;

    cm = img.getColorModel();
    isAlphaPremultiplied = cm.isAlphaPremultiplied();
    raster = img.copyData(null);

    return new BufferedImage(cm, raster, isAlphaPremultiplied, null);
  }

  /**
   * Returns all the pixels of the image as an int array (row-wise).
   *
   * @param img		the image to get the pixels from
   * @return		the pixel array
   * @see		BufferedImage#getRGB(int, int)
   */
  public static int[] getPixels(BufferedImage img) {
    int[]	result;
    int		y;
    int		x;
    int		i;

    result = new int[img.getWidth() * img.getHeight()];
    i      = 0;
    for (y = 0; y < img.getHeight(); y++) {
      for (x = 0; x < img.getWidth(); x++) {
	result[i] = img.getRGB(x, y);
	i++;
      }
    }

    return result;
  }

  /**
   * Returns the pixels of the image as an 2-D int array (row-wise).
   *
   * @param img		the image to get the pixels from
   * @return		the pixel array
   * @see		BufferedImage#getRGB(int, int)
   */
  public static int[][] getPixelRaster(BufferedImage img) {
    int[][]	result;
    int		y;
    int		x;

    result = new int[img.getHeight()][img.getWidth()];
    for (y = 0; y < img.getHeight(); y++) {
      for (x = 0; x < img.getWidth(); x++)
	result[y][x] = img.getRGB(x, y);
    }

    return result;
  }
}
