/*
 * MovieWriter.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 * Copyright (C) 1999-2011 Sun Microsystems
 */

package adams.flow.sink;

import java.awt.Dimension;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Vector;

import javax.media.Buffer;
import javax.media.ConfigureCompleteEvent;
import javax.media.ControllerEvent;
import javax.media.ControllerListener;
import javax.media.DataSink;
import javax.media.EndOfMediaEvent;
import javax.media.Format;
import javax.media.Manager;
import javax.media.MediaLocator;
import javax.media.PrefetchCompleteEvent;
import javax.media.Processor;
import javax.media.RealizeCompleteEvent;
import javax.media.ResourceUnavailableEvent;
import javax.media.Time;
import javax.media.control.TrackControl;
import javax.media.datasink.DataSinkErrorEvent;
import javax.media.datasink.DataSinkEvent;
import javax.media.datasink.DataSinkListener;
import javax.media.datasink.EndOfStreamEvent;
import javax.media.format.VideoFormat;
import javax.media.protocol.ContentDescriptor;
import javax.media.protocol.DataSource;
import javax.media.protocol.FileTypeDescriptor;
import javax.media.protocol.PullBufferDataSource;
import javax.media.protocol.PullBufferStream;

import adams.core.io.PlaceholderFile;

/**
 <!-- globalinfo-start -->
 * Actor for turning JPEG images into Quicktime movies.<br/>
 * <br/>
 * Original code taken from Sun's JpegImagesToMovie.java example:<br/>
 * http:&#47;&#47;www.koders.com&#47;java&#47;fid8AE9208DC5FB846910DFD3F935D64E7C9A5BB784.aspx
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String[]<br/>
 * &nbsp;&nbsp;&nbsp;java.io.File[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MovieWriter
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-output &lt;adams.core.io.PlaceholderFile&gt; (property: outputFile)
 * &nbsp;&nbsp;&nbsp;The filename of the image to write.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 * 
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The screen width of the movie.
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The screen height of the movie.
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 * <pre>-frame-rate &lt;int&gt; (property: frameRate)
 * &nbsp;&nbsp;&nbsp;The frame rate of the movie.
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4380 $
 */
public class MovieWriter
  extends AbstractFileWriter 
  implements ControllerListener, DataSinkListener {

  /** for serialization. */
  private static final long serialVersionUID = 7509908838736709270L;
  
  /**
   * A DataSource to read from a list of JPEG image files and
   * turn that into a stream of JMF buffers.
   * The DataSource is not seekable or positionable.
   */
  class ImageDataSource extends PullBufferDataSource {
    ImageSourceStream streams[];

    ImageDataSource(int width, int height, int frameRate, Vector images) {
      streams = new ImageSourceStream[1];
      streams[0] = new ImageSourceStream(width, height, frameRate, images);
    }

    public void setLocator(MediaLocator source) {
    }

    public MediaLocator getLocator() {
      return null;
    }

    /**
     * Content type is of RAW since we are sending buffers of video
     * frames without a container format.
     */
    public String getContentType() {
      return ContentDescriptor.RAW;
    }

    public void connect() {
    }

    public void disconnect() {
    }

    public void start() {
    }

    public void stop() {
    }

    /**
     * Return the ImageSourceStreams.
     */
    public PullBufferStream[] getStreams() {
      return streams;
    }

    /**
     * We could have derived the duration from the number of
     * frames and frame rate.  But for the purpose of this program,
     * it's not necessary.
     */
    public Time getDuration() {
      return DURATION_UNKNOWN;
    }

    public Object[] getControls() {
      return new Object[0];
    }

    public Object getControl(String type) {
      return null;
    }
  }

  /**
   * The source stream to go along with ImageDataSource.
   */
  class ImageSourceStream implements PullBufferStream {

    Vector images;
    int width, height;
    VideoFormat format;

    int nextImage = 0;  // index of the next image to be read.
    boolean ended = false;

    public ImageSourceStream(int width, int height, int frameRate, Vector images) {
      this.width = width;
      this.height = height;
      this.images = images;

      format = new VideoFormat(VideoFormat.JPEG,
	  new Dimension(width, height),
	  Format.NOT_SPECIFIED,
	  Format.byteArray,
	  (float)frameRate);
    }

    /**
     * We should never need to block assuming data are read from files.
     */
    public boolean willReadBlock() {
      return false;
    }

    /**
     * This is called from the Processor to read a frame worth
     * of video data.
     */
    public void read(Buffer buf) throws IOException {
      // Check if we've finished all the frames.
      if (nextImage >= images.size()) {
	buf.setEOM(true);
	buf.setOffset(0);
	buf.setLength(0);
	ended = true;
	return;
      }

      String imageFile = (String)images.elementAt(nextImage);
      nextImage++;

      // Open a random access file for the next image. 
      RandomAccessFile raFile;
      raFile = new RandomAccessFile(imageFile, "r");

      byte data[] = null;

      // Check the input buffer type & size.

      if (buf.getData() instanceof byte[])
	data = (byte[])buf.getData();

      // Check to see the given buffer is big enough for the frame.
      if (data == null || data.length < raFile.length()) {
	data = new byte[(int)raFile.length()];
	buf.setData(data);
      }

      // Read the entire JPEG image from the file.
      raFile.readFully(data, 0, (int)raFile.length());

      buf.setOffset(0);
      buf.setLength((int)raFile.length());
      buf.setFormat(format);
      buf.setFlags(buf.getFlags() | Buffer.FLAG_KEY_FRAME);

      // Close the random access file.
      raFile.close();
    }

    /**
     * Return the format of each video frame.  That will be JPEG.
     */
    public Format getFormat() {
      return format;
    }

    public ContentDescriptor getContentDescriptor() {
      return new ContentDescriptor(ContentDescriptor.RAW);
    }

    public long getContentLength() {
      return 0;
    }

    public boolean endOfStream() {
      return ended;
    }

    public Object[] getControls() {
      return new Object[0];
    }

    public Object getControl(String type) {
      return null;
    }
  }
  
  /** the width. */
  protected int m_Width;
  
  /** the height. */
  protected int m_Height;
  
  /** the frame rate. */
  protected int m_FrameRate;

  protected transient Object m_WaitSync;
  protected boolean m_StateTransitionOK;

  protected transient Object m_WaitFileSync;
  protected boolean m_FileDone;
  protected boolean m_FileSuccess;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return 
	"Actor for turning JPEG images into Quicktime movies.\n\n"
	+ "Original code taken from Sun's JpegImagesToMovie.java example:\n"
	+ "http://www.koders.com/java/fid8AE9208DC5FB846910DFD3F935D64E7C9A5BB784.aspx";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "width", "width",
	    -1, -1, null);

    m_OptionManager.add(
	    "height", "height",
	    -1, -1, null);

    m_OptionManager.add(
	    "frame-rate", "frameRate",
	    -1, -1, null);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;
    
    variable = getOptionManager().getVariableForProperty("width");
    if (variable != null)
      result = variable;
    else
      result = "" + m_Width;
    
    result += " x ";
    variable = getOptionManager().getVariableForProperty("height");
    if (variable != null)
      result += variable;
    else
      result += "" + m_Height;
    
    result += " @ ";
    variable = getOptionManager().getVariableForProperty("frameRate");
    if (variable != null)
      result += variable;
    else
      result += "" + m_FrameRate;

    return result;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputFileTipText() {
    return "The filename of the image to write.";
  }

  /**
   * Sets the screen width of the movie.
   *
   * @param value 	the width
   */
  public void setWidth(int value) {
    m_Width = value;
    reset();
  }

  /**
   * Returns the screen width of the movie.
   *
   * @return 		the width
   */
  public int getWidth() {
    return m_Width;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String widthTipText() {
    return "The screen width of the movie.";
  }

  /**
   * Sets the screen height of the movie.
   *
   * @param value 	the height
   */
  public void setHeight(int value) {
    m_Height = value;
    reset();
  }

  /**
   * Returns the screen height of the movie.
   *
   * @return 		the height
   */
  public int getHeight() {
    return m_Height;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String heightTipText() {
    return "The screen height of the movie.";
  }

  /**
   * Sets the frame rate of the movie.
   *
   * @param value 	the frame rate
   */
  public void setFrameRate(int value) {
    m_FrameRate = value;
    reset();
  }

  /**
   * Returns the frame rate of the movie.
   *
   * @return 		the frame rate
   */
  public int getFrameRate() {
    return m_FrameRate;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String frameRateTipText() {
    return "The frame rate of the movie.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String[].class, java.io.File[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String[].class, File[].class};
  }
  
  /**
   * Generates the movie.
   * 
   * @param outML	the media locator
   * @param inFiles	the file to create the movie from
   * @return		null if successful, otherwise error message
   */
  protected String create(MediaLocator outML, Vector<String> inFiles) {
    ImageDataSource ids = new ImageDataSource(m_Width, m_Height, m_FrameRate, inFiles);

    Processor p;

    try {
      p = Manager.createProcessor(ids);
    } 
    catch (Exception e) {
      return "Cannot create a processor from the data source.";
    }

    p.addControllerListener(this);

    // Put the Processor into configured state so we can set
    // some processing options on the processor.
    p.configure();
    if (!waitForState(p, Processor.Configured))
      return "Failed to configure the processor.";

    // Set the output content descriptor to QuickTime. 
    p.setContentDescriptor(new ContentDescriptor(FileTypeDescriptor.QUICKTIME));

    // Query for the processor for supported formats.
    // Then set it on the processor.
    TrackControl tcs[] = p.getTrackControls();
    Format f[] = tcs[0].getSupportedFormats();
    if ((f == null) || (f.length <= 0))
      return "The mux does not support the input format: " + tcs[0].getFormat();

    tcs[0].setFormat(f[0]);

    // We are done with programming the processor.  Let's just
    // realize it.
    p.realize();
    if (!waitForState(p, Processor.Realized))
      return "Failed to realize the processor.";

    // Now, we'll need to create a DataSink.
    DataSink dsink;
    if ((dsink = createDataSink(p, outML)) == null)
      return "Failed to create a DataSink for the given output MediaLocator: " + outML;

    dsink.addDataSinkListener(this);
    m_FileDone = false;

    // OK, we can now start the actual transcoding.
    try {
      p.start();
      dsink.start();
    } 
    catch (IOException e) {
      return "IO error during processing";
    }

    // Wait for EndOfStream event.
    waitForFileDone();

    // Cleanup.
    try {
      dsink.close();
    } 
    catch (Exception e) {
      // ignored
    }
    p.removeControllerListener(this);

    return null;
  }


  /**
   * Create the DataSink.
   */
  protected DataSink createDataSink(Processor p, MediaLocator outML) {

    DataSource ds;

    if ((ds = p.getDataOutput()) == null) {
      getSystemErr().println("Something is really wrong: the processor does not have an output DataSource");
      return null;
    }

    DataSink dsink;

    try {
      dsink = Manager.createDataSink(ds, outML);
      dsink.open();
    } catch (Exception e) {
      getSystemErr().println("Cannot create the DataSink: " + e);
      return null;
    }

    return dsink;
  }


  /**
   * Block until the processor has transitioned to the given state.
   * Return false if the transition failed.
   */
  protected boolean waitForState(Processor p, int state) {
    synchronized (m_WaitSync) {
      try {
	while (p.getState() < state && m_StateTransitionOK)
	  m_WaitSync.wait();
      } 
      catch (Exception e) {
	// ignored
      }
    }
    return m_StateTransitionOK;
  }


  /**
   * Controller Listener.
   */
  public void controllerUpdate(ControllerEvent evt) {
    if (evt instanceof ConfigureCompleteEvent ||
	evt instanceof RealizeCompleteEvent ||
	evt instanceof PrefetchCompleteEvent) {
      synchronized(m_WaitSync) {
	m_StateTransitionOK = true;
	m_WaitSync.notifyAll();
      }
    } 
    else if (evt instanceof ResourceUnavailableEvent) {
      synchronized(m_WaitSync) {
	m_StateTransitionOK = false;
	m_WaitSync.notifyAll();
      }
    } 
    else if (evt instanceof EndOfMediaEvent) {
      evt.getSourceController().stop();
      evt.getSourceController().close();
    }
  }

  /**
   * Block until file writing is done. 
   */
  boolean waitForFileDone() {
    synchronized(m_WaitFileSync) {
      try {
	while (!m_FileDone)
	  m_WaitFileSync.wait();
      } 
      catch (Exception e) {
	// ignored
      }
    }
    return m_FileSuccess;
  }

  /**
   * Event handler for the file writer.
   */
  public void dataSinkUpdate(DataSinkEvent evt) {
    if (evt instanceof EndOfStreamEvent) {
      synchronized(m_WaitFileSync) {
	m_FileDone = true;
	m_WaitFileSync.notifyAll();
      }
    } 
    else if (evt instanceof DataSinkErrorEvent) {
      synchronized(m_WaitFileSync) {
	m_FileDone = true;
	m_FileSuccess = false;
	m_WaitFileSync.notifyAll();
      }
    }
  }  
  
  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    MediaLocator	locator;
    Vector<String>	files;
    File[]		fileFiles;
    String[]		strFiles;

    result = null;

    m_WaitSync          = new Object();
    m_WaitFileSync      = new Object();
    m_StateTransitionOK = true;
    m_FileDone          = false;
    m_FileSuccess       = true;
    
    if (result == null) {
      locator = new MediaLocator("file:" + m_OutputFile.getAbsolutePath());
      files   = new Vector<String>();
      if (m_InputToken.getPayload() instanceof File[]) {
	fileFiles = (File[]) m_InputToken.getPayload();
	for (File file: fileFiles)
	  files.add(file.getAbsolutePath());
      }
      else {
	strFiles = (String[]) m_InputToken.getPayload();
	for (String file: strFiles)
	  files.add(new PlaceholderFile(file).getAbsolutePath());
      }
      result = create(locator, files);
    }

    return result;
  }
}
