/*
 *    Jython.java
 *    Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.flow.condition;

import java.util.Hashtable;

import adams.flow.core.AdditionalOptionsHandler;
import adams.flow.core.AdditionalOptionsHandlerUtils;

/**
 <!-- globalinfo-start -->
 * A condition that uses the condition defined in an external Jython script.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-script &lt;adams.core.io.PlaceholderFile&gt; (property: scriptFile)
 * &nbsp;&nbsp;&nbsp;The script file to load and execute.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-options &lt;java.lang.String&gt; (property: scriptOptions)
 * &nbsp;&nbsp;&nbsp;The options for the Jython script; must consist of 'key=value' pairs separated
 * &nbsp;&nbsp;&nbsp;by blanks; the value of 'key' can be accessed via the 'getAdditionalOption
 * &nbsp;&nbsp;&nbsp;(String)' in the Jython actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2105 $
 * @see adams.core.scripting.Jython
 */
public class Jython
  extends AbstractScriptCondition {

  /** for serialization. */
  private static final long serialVersionUID = -2562159780167388413L;

  /** the loaded script object. */
  protected transient AbstractCondition m_ConditionObject;

  /** for storing the additional options. */
  protected Hashtable<String,String> m_AdditionalOptions;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "A condition that uses the condition defined in an external Jython script.";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_AdditionalOptions = new Hashtable<String,String>();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptOptionsTipText() {
    return
        "The options for the Jython script; must consist of 'key=value' pairs "
      + "separated by blanks; the value of 'key' can be accessed via the "
      + "'getAdditionalOption(String)' in the Jython actor.";
  }

  /**
   * Loads the scripts object and sets its options.
   *
   * @return		null if OK, otherwise the error message
   */
  protected String loadScriptObject() {
    String	result;

    result = null;

    try {
      m_ScriptObject = null;
      if (m_ScriptFile.isFile()) {
	m_ScriptObject = (AbstractCondition) adams.core.scripting.Jython.newInstance(m_ScriptFile, AbstractCondition.class);
	if (m_ScriptObject != null) {
	  if (!AdditionalOptionsHandlerUtils.setOptions(m_ScriptObject, m_ScriptOptions))
	    result = "Does not implement '" + AdditionalOptionsHandler.class.getName() + "': " + m_ScriptFile;
	}
	else {
	  result = "Failed to instantiate script '" + m_ScriptFile + "'!";
	}
      }
      else {
	result = "No script provided!";
      }
    }
    catch (Exception e) {
      m_ScriptObject = null;
      getSystemErr().printStackTrace(e);
      result = "Failed to initialize Jython script '" + m_ScriptFile + "': " + e.toString();
    }

    return result;
  }

  /**
   * Checks the script object.
   *
   * @return		always null, i.e., OK
   */
  protected String checkScriptObject() {
    m_ConditionObject = (AbstractCondition) m_ScriptObject;
    return null;
  }

  /**
   * Runs the script test.
   *
   * @return		the test result, null if everything OK, otherwise
   * 			the error message
   */
  protected String performScriptTest() {
    return m_ConditionObject.getTestResult();
  }

  /**
   * Sets the additional options.
   *
   * @param options	the options (name &lt;-&gt;value relation)
   */
  public void setAdditionalOptions(Hashtable<String,String> options) {
    m_AdditionalOptions = (Hashtable<String,String>) options.clone();
  }

  /**
   * Returns the value associated with the (additional) option.
   *
   * @param name	the name of the additional option to retrieve
   * @param defValue	the default value
   * @return		the value or null if not available
   */
  public String getAdditionalOption(String name, String defValue) {
    if (m_AdditionalOptions.containsKey(name))
      return m_AdditionalOptions.get(name);
    else
      return defValue;
  }
}
