/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MOAClustererEvaluation.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Hashtable;

import moa.clusterers.AbstractClusterer;
import moa.core.Measurement;
import moa.evaluation.BasicClusteringPerformanceEvaluator;
import moa.evaluation.LearningPerformanceEvaluator;
import moa.options.ClassOption;
import weka.core.Instance;
import weka.core.MOAUtils;
import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.source.GlobalSource;

/**
 <!-- globalinfo-start -->
 * Evaluates a MOA clusterer using prequential evaluation. With each incoming instance, the clusterer is first evaluated, then trained.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;moa.core.Measurement[]<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MOAClustererEvaluation
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-clusterer &lt;java.lang.String&gt; (property: clusterer)
 * &nbsp;&nbsp;&nbsp;The name of the global MOA clusterer to train&#47;evaluate.
 * &nbsp;&nbsp;&nbsp;default: MOAClusterer
 * </pre>
 *
 * <pre>-evaluator &lt;moa.options.ClassOption&gt; (property: evaluator)
 * &nbsp;&nbsp;&nbsp;The MOA evaluator to use for evaluating a trained MOA clusterer.
 * &nbsp;&nbsp;&nbsp;default: moa.evaluation.BasicClusteringPerformanceEvaluator
 * </pre>
 *
 * <pre>-output-interval &lt;int&gt; (property: outputInterval)
 * &nbsp;&nbsp;&nbsp;The number of tokens to skip before evaluating the clusterer stored in the
 * &nbsp;&nbsp;&nbsp;token.
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class MOAClustererEvaluation
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 1410487605033307517L;

  /** the key for storing the current clusterer in the backup. */
  public final static String BACKUP_CLUSTERER = "clusterer";

  /** the name of the global clusterer to use. */
  protected GlobalActorReference m_Clusterer;

  /** the model to use for prediction/training. */
  protected AbstractClusterer m_ActualClusterer;

  /** the evaluation to use. */
  protected moa.options.ClassOption m_Evaluator;

  /** the actual evaluator to use. */
  protected moa.evaluation.LearningPerformanceEvaluator m_ActualEvaluator;

  /** the output interval. */
  protected int m_OutputInterval;

  /** the current count of tokens that have passed through this actor. */
  protected int m_Count;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates a MOA clusterer using prequential evaluation. With each "
      + "incoming instance, the clusterer is first evaluated, then trained.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "clusterer", "clusterer",
	    new GlobalActorReference("MOAClusterer"));

    m_OptionManager.add(
	    "evaluator", "evaluator",
	    getDefaultOption());

    m_OptionManager.add(
	    "output-interval", "outputInterval",
	    1, 1, null);
  }

  /**
   * Initializes the members.
   */
  protected void reset() {
    super.reset();

    m_ActualEvaluator = null;
    m_Count           = 0;
    m_ActualClusterer = null;
  }

  /**
   * Sets the global clusterer to use.
   *
   * @param value	the clusterer name
   */
  public void setClusterer(GlobalActorReference value) {
    m_Clusterer = value;
    reset();
  }

  /**
   * Returns the global clusterer to use.
   *
   * @return		the clusterer name
   */
  public GlobalActorReference getClusterer() {
    return m_Clusterer;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String clustererTipText() {
    return "The name of the global MOA clusterer to train/evaluate.";
  }

  /**
   * Returns the default evaluator.
   *
   * @return		the evaluator
   */
  protected moa.evaluation.LearningPerformanceEvaluator getDefaultEvaluator() {
    return new BasicClusteringPerformanceEvaluator();
  }

  /**
   * Returns the default class option.
   *
   * @return		the option
   */
  protected ClassOption getDefaultOption() {
    return new ClassOption(
	"evaluator",
	'e',
	"The MOA clusterer performance evaluator to use from within ADAMS.",
	moa.evaluation.LearningPerformanceEvaluator.class,
	getDefaultEvaluator().getClass().getName().replace("moa.evaluation.", ""),
	getDefaultEvaluator().getClass().getName());
  }

  /**
   * Sets the evaluator to use.
   *
   * @param value	the evaluator
   */
  public void setEvaluator(ClassOption value) {
    m_Evaluator = (ClassOption) value.copy();
    reset();
  }

  /**
   * Returns the evaluator in use.
   *
   * @return		the evaluator
   */
  public ClassOption getEvaluator() {
    return m_Evaluator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String evaluatorTipText() {
    return "The MOA evaluator to use for evaluating a trained MOA clusterer.";
  }

  /**
   * Returns the current evaluator, based on the class option.
   *
   * @return		the evaluator
   * @see		#getEvaluator()
   */
  protected LearningPerformanceEvaluator getCurrentEvaluator() {
    return (moa.evaluation.LearningPerformanceEvaluator) MOAUtils.fromOption(m_Evaluator);
  }

  /**
   * Sets the number of tokens after which to evaluate the clusterer.
   *
   * @param value	the interval
   */
  public void setOutputInterval(int value) {
    m_OutputInterval = value;
    reset();
  }

  /**
   * Returns the number of tokens after which to evaluate the clusterer.
   *
   * @return		the interval
   */
  public int getOutputInterval() {
    return m_OutputInterval;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputIntervalTipText() {
    return "The number of tokens to skip before evaluating the clusterer stored in the token.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("clusterer");
    if (variable != null)
      result = variable;
    else
      result = m_Clusterer.toString();

    variable = getOptionManager().getVariableForProperty("evaluator");
    if (variable != null)
      result += ", " + variable;
    else
      result += ", " + getCurrentEvaluator().getClass().getName().replace("moa.evaluation.", "");

    variable = getOptionManager().getVariableForProperty("outputInterval");
    if (variable != null)
      result += "/" + variable;
    else
      result += "/" + ((m_OutputInterval == 1) ? "always" : m_OutputInterval);

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.core.Instance.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Instance.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->moa.core.Measurement[].class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Measurement[].class};
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_CLUSTERER);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_ActualClusterer != null)
      result.put(BACKUP_CLUSTERER, m_ActualClusterer);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_CLUSTERER)) {
      m_ActualClusterer = (AbstractClusterer) state.get(BACKUP_CLUSTERER);
      state.remove(BACKUP_CLUSTERER);
    }

    super.restoreState(state);
  }

  /**
   * Returns an instance of the global clusterer.
   *
   * @return		the clusterer
   */
  protected AbstractClusterer getClustererInstance() {
    GlobalSource	global;
    AbstractClusterer	result;
    AbstractActor	actor;

    result = null;

    global = new GlobalSource();
    global.setGlobalName(m_Clusterer);
    global.setParent(getParent());
    if (global.setUp() == null) {
      global.execute();
      result = (AbstractClusterer) global.output().getPayload();
      actor  = global.getGlobalActor();
      global.wrapUp();
      global.cleanUp();
      // fix wrapUp/cleanUp of global actor, e.g., re-registering variable listeners
      if (actor != null)
	actor.setUp();
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    Instance		inst;
    Instance		testInst;
    double[] 		prediction;

    result = null;

    if (m_ActualEvaluator == null)
      m_ActualEvaluator = (moa.evaluation.LearningPerformanceEvaluator) MOAUtils.fromOption(m_Evaluator);
    inst = (Instance) m_InputToken.getPayload();
    if (m_ActualClusterer == null) {
      m_ActualClusterer = getClustererInstance();
      if (m_ActualClusterer == null) {
	result = "Failed to located clusterer '" + m_Clusterer + "'!";
	return result;
      }
    }

    // test
    testInst   = (Instance) inst.copy();
    prediction = m_ActualClusterer.getVotesForInstance(testInst);
    m_ActualEvaluator.addLearningAttempt(-1, prediction, testInst.weight());  // TODO: class necessary???

    // train
    m_ActualClusterer.trainOnInstance(inst);

    m_Count++;
    if (m_Count % m_OutputInterval == 0) {
      m_Count = 0;
      m_OutputToken = new Token(m_ActualEvaluator.getPerformanceMeasurements());
      updateProvenance(m_OutputToken);
    }

    return result;
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled()) {
      cont.setProvenance(m_InputToken.getProvenance());
      cont.addProvenance(new ProvenanceInformation(ActorType.EVALUATOR, m_InputToken.getPayload().getClass(), this, m_OutputToken.getPayload().getClass()));
    }
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    super.wrapUp();

    m_ActualEvaluator = null;
    m_ActualClusterer = null;
  }
}
