/**
 * ODFSpreadSheetWriter.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.OutputStream;

import javax.swing.table.DefaultTableModel;

import adams.core.io.SpreadSheet.Cell;
import adams.core.io.SpreadSheet.Row;

/**
 <!-- globalinfo-start -->
 * Writes ODF (Open Document Format) spreadsheet files.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-missing &lt;java.lang.String&gt; (property: missingValue)
 * &nbsp;&nbsp;&nbsp;The placeholder for missing values.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3896 $
 */
public class ODFSpreadSheetWriter
  extends AbstractSpreadSheetWriter {

  /** for serialization. */
  private static final long serialVersionUID = -3549185519778801930L;

  /** the binary file extension. */
  public static String FILE_EXTENSION = ".ods";

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Writes ODF (Open Document Format) spreadsheet files.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "OpenDocument format";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{"ods"};
  }

  /**
   * Returns whether to write to an OutputStream rather than a Writer.
   *
   * @return		true if to write to an OutputStream
   */
  protected boolean getUseOutputStream() {
    return false;
  }

  /**
   * Performs the actual writing. The caller must ensure that the writer gets
   * closed.
   *
   * @param content	the spreadsheet to write
   * @param out		the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, OutputStream out) {
    boolean		result;
    org.jopendocument.dom.spreadsheet.SpreadSheet	spreadsheet;
    int			i;
    int			n;
    DefaultTableModel	model;
    Row			row;
    Cell		cell;

    result = true;

    try {
      model = new DefaultTableModel(content.getRowCount() + 1, content.getColumnCount());  // +1 for header
      // header
      for (i = 0; i < content.getColumnCount(); i++)
	model.setValueAt(content.getHeaderRow().getCell(i).getContent(), 0, i);
      // data
      for (n = 0; n < content.getRowCount(); n++) {
	row = content.getRow(n);
	for (i = 0; i < content.getColumnCount(); i++) {
	  cell = row.getCell(i);
	  if (cell.isMissing())
	    model.setValueAt(m_MissingValue, n + 1, i);
	  else
	    model.setValueAt(cell.getContent(), n + 1, i);
	}
      }
      spreadsheet = org.jopendocument.dom.spreadsheet.SpreadSheet.createEmpty(model);
      spreadsheet.getPackage().save(out);
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }
}
