/**
 * ExportAsSpreadsheet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.sendto;

import java.io.File;

import javax.swing.JTable;

import adams.core.io.AbstractSpreadSheetWriter;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.gui.chooser.SpreadSheetFileChooser;

/**
 * Action for exporting a table as spreadsheet.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3897 $
 */
public class ExportAsSpreadsheet
  extends AbstractSendToAction {

  /** for serialization. */
  private static final long serialVersionUID = -5286281737195775697L;

  /**
   * Returns the short description of the sendto action.
   * Description gets used for menu items.
   *
   * @return		the short description
   */
  public String getAction() {
    return "Export as spreadsheet";
  }

  /**
   * Returns the file name of the icon.
   *
   * @return		the filename or null if no icon available
   */
  public String getIconName() {
    return "spreadsheet.png";
  }

  /**
   * Returns the classes that the action accepts.
   *
   * @return		the accepted classes
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class, JTable.class};
  }

  /**
   * Performs the actual sending/forwarding/processing of the data.
   *
   * @param o		the object to send
   * @return		null if everything OK, otherwise error message
   */
  public String send(Object o) {
    String			result;
    int				retVal;
    SpreadSheet			sheet;
    SpreadSheetFileChooser 	fileChooser;
    File			file;
    AbstractSpreadSheetWriter	writer;
    String			msg;
    int				i;
    int				n;
    JTable			table;
    Row				row;
    Object			value;

    result = null;

    sheet = null;
    if (o instanceof SpreadSheet) {
      sheet = (SpreadSheet) o;
    }
    else if (o instanceof JTable) {
      sheet = new SpreadSheet();
      table = (JTable) o;
      // header
      row = sheet.getHeaderRow();
      for (i = 0; i < table.getColumnCount(); i++)
	row.addCell("" + i).setContent(table.getColumnName(i));
      for (n = 0; n < table.getRowCount(); n++) {
	row = sheet.addRow("" + n);
	for (i = 0; i < table.getColumnCount(); i++) {
	  value = table.getValueAt(n, i);
	  if (value != null)
	    row.addCell("" + i).setContent(value.toString());
	}
      }
    }

    if (sheet != null) {
      fileChooser = new SpreadSheetFileChooser();
      // display save dialog
      retVal = fileChooser.showSaveDialog(null);
      if (retVal != SpreadSheetFileChooser.APPROVE_OPTION)
        result = "Export canceled by user!";

      if (result == null) {
	// save the file
	try {
	  file   = fileChooser.getSelectedFile().getAbsoluteFile();
	  writer = fileChooser.getWriter();
	  if (!writer.write(sheet, file))
	    result = "Failed to export spreadsheet!";
	}
	catch (Exception e) {
	  msg    = "Failed to export spreadsheet: ";
	  result = msg + e;
	  getSystemErr().println(msg);
	  getSystemErr().printStackTrace(e);
	}
      }
    }
    else {
      result = "Cannot export object as spreadsheet: " + o.getClass();
    }

    return result;
  }
}
