/*
 * SpreadSheetCell.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.core.Index;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Extracts a single value from a spreadsheet.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetCell
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-row &lt;java.lang.String&gt; (property: row)
 * &nbsp;&nbsp;&nbsp;The row of the cell to retrieve; An index is a number starting with 1; the
 * &nbsp;&nbsp;&nbsp;following placeholders can be used as well: first, second, third, last_2,
 * &nbsp;&nbsp;&nbsp; last_1, last
 * &nbsp;&nbsp;&nbsp;default: 1
 * </pre>
 *
 * <pre>-col &lt;java.lang.String&gt; (property: column)
 * &nbsp;&nbsp;&nbsp;The column of the cell to retrieve; An index is a number starting with 1;
 * &nbsp;&nbsp;&nbsp; the following placeholders can be used as well: first, second, third, last
 * &nbsp;&nbsp;&nbsp;_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2415 $
 */
public class SpreadSheetCell
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -253714973019682939L;

  /** the row of the cell to obtain. */
  protected Index m_Row;

  /** the column of the cell to obtain. */
  protected Index m_Column;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Extracts a single value from a spreadsheet.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "row", "row",
	    "1");

    m_OptionManager.add(
	    "col", "column",
	    "1");
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Row    = new Index();
    m_Column = new Index();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variableRow;
    String	variableCol;

    variableRow = getOptionManager().getVariableForProperty("row");
    variableCol = getOptionManager().getVariableForProperty("column");

    result = "row: ";
    if (variableRow != null)
      result += variableRow;
    else
      result += m_Row.getIndex();

    result += "/";

    result += "col: ";
    if (variableCol != null)
      result += variableCol;
    else
      result += m_Column.getIndex();

    return result;
  }

  /**
   * Sets the row of the cell.
   *
   * @param value	the row
   */
  public void setRow(String value) {
    m_Row.setIndex(value);
    reset();
  }

  /**
   * Returns the row of the cell.
   *
   * @return		the row
   */
  public String getRow() {
    return m_Row.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String rowTipText() {
    return "The row of the cell to retrieve; " + m_Row.getExample();
  }

  /**
   * Sets the column of the cell.
   *
   * @param value	the column
   */
  public void setColumn(String value) {
    m_Column.setIndex(value);
    reset();
  }

  /**
   * Returns the column of the cell.
   *
   * @return		the column
   */
  public String getColumn() {
    return m_Column.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String columnTipText() {
    return "The column of the cell to retrieve; " + m_Column.getExample();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.core.io.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    SpreadSheet	sheet;
    Row		row;
    String	key;
    String	value;

    result = null;

    sheet = (SpreadSheet) m_InputToken.getPayload();
    m_Row.setMax(sheet.getRowCount());
    m_Column.setMax(sheet.getColumnCount());

    if (m_Row.getIntIndex() == -1) {
      result = "Illegal row index ('" + m_Row.getIndex() + "')?";
    }
    else if (m_Column.getIntIndex() == -1) {
      result = "Illegal column index ('" + m_Column.getIndex() + "')?";
    }
    else {
      row   = sheet.getRow(m_Row.getIntIndex());
      key   = row.getCellKey(m_Column.getIntIndex());
      if (row.getCell(key).isMissing())
	value = SpreadSheet.MISSING_VALUE;
      else
	value = row.getCell(key).getContent();
      m_OutputToken = new Token(value);
    }

    return result;
  }
}
