/*
 * SpreadSheetInfo.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Arrays;
import java.util.HashSet;

import adams.core.Index;
import adams.core.io.SpreadSheet;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Outputs statistics of a SpreadSheet object.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetInfo
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-type &lt;NUM_COLUMNS|NUM_ROWS|COLUMN_NAME&gt; (property: type)
 * &nbsp;&nbsp;&nbsp;The type of information to generate.
 * &nbsp;&nbsp;&nbsp;default: NUM_ROWS
 * </pre>
 *
 * <pre>-column-index &lt;java.lang.String&gt; (property: columnIndex)
 * &nbsp;&nbsp;&nbsp;The column index to use for generating column-specific information. An index
 * &nbsp;&nbsp;&nbsp;is a number starting with 1; the following placeholders can be used as well:
 * &nbsp;&nbsp;&nbsp; first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: last
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3681 $
 */
public class SpreadSheetInfo
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -3019442578354930841L;

  /**
   * The type of information to generate.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3681 $
   */
  public enum InfoType {
    /** the number of columns. */
    NUM_COLUMNS,
    /** the number of rows. */
    NUM_ROWS,
    /** the name of the column (at specified index). */
    COLUMN_NAME
  }

  /** the type of information to generate. */
  protected InfoType m_Type;

  /** the index of the column to get the information for. */
  protected Index m_ColumnIndex;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Outputs statistics of a SpreadSheet object.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "type", "type",
	    InfoType.NUM_ROWS);

    m_OptionManager.add(
	    "column-index", "columnIndex",
	    "last");
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_ColumnIndex = new Index();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String		result;
    String		variable;
    HashSet<InfoType>	types;

    result = null;

    variable = getOptionManager().getVariableForProperty("type");
    if (variable != null)
      result = variable;
    else
      result = "" + m_Type;

    types = new HashSet<InfoType>(
	Arrays.asList(
	    new InfoType[]{
		InfoType.NUM_COLUMNS,
		InfoType.NUM_ROWS,
	    }));
    if (!types.contains(m_Type)) {
      result += ", index: ";
      variable = getOptionManager().getVariableForProperty("columnIndex");
      if (variable != null)
        result += variable;
      else
        result += m_ColumnIndex.getIndex();
    }

    return result;
  }

  /**
   * Sets the type of information to generate.
   *
   * @param value	the type
   */
  public void setType(InfoType value) {
    m_Type = value;
    reset();
  }

  /**
   * Returns the type of information to generate.
   *
   * @return		the type
   */
  public InfoType getType() {
    return m_Type;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String typeTipText() {
    return "The type of information to generate.";
  }

  /**
   * Sets the column index to use for column specific information.
   *
   * @param value	the 1-based index
   */
  public void setColumnIndex(String value) {
    m_ColumnIndex.setIndex(value);
    reset();
  }

  /**
   * Returns the column index to use for column specific information.
   *
   * @return		the 1-based index
   */
  public String getColumnIndex() {
    return m_ColumnIndex.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String columnIndexTipText() {
    return "The column index to use for generating column-specific information. " + m_ColumnIndex.getExample();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.core.io.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Integer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    switch (m_Type) {
      case COLUMN_NAME:
	return new Class[]{String.class};

      case NUM_COLUMNS:
      case NUM_ROWS:
	return new Class[]{Integer.class};

      default:
	throw new IllegalStateException("Unhandled info type: " + m_Type);
    }
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    SpreadSheet	sheet;
    int		index;

    result = null;

    sheet = (SpreadSheet) m_InputToken.getPayload();
    m_ColumnIndex.setMax(sheet.getColumnCount());
    index = m_ColumnIndex.getIntIndex();

    switch (m_Type) {
      case COLUMN_NAME:
	if (index != -1)
	  m_OutputToken = new Token(sheet.getHeaderRow().getCell(index).getContent());
	break;

      case NUM_COLUMNS:
	m_OutputToken = new Token(sheet.getColumnCount());
	break;

      case NUM_ROWS:
	m_OutputToken = new Token(sheet.getRowCount());
	break;

      default:
	result = "Unhandled info type: " + m_Type;
    }

    return result;
  }
}
