/**
 * SpreadSheetViewerPanel.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.tools;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.Properties;
import adams.core.io.AbstractSpreadSheetReader;
import adams.core.io.CsvSpreadSheetReader;
import adams.core.io.CsvSpreadSheetWriter;
import adams.core.io.PlaceholderFile;
import adams.core.io.SpreadSheet;
import adams.env.Environment;
import adams.env.SpreadSheetViewerPanelDefinition;
import adams.gui.chooser.SpreadSheetFileChooser;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseTabbedPane;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.core.RecentFilesHandler;
import adams.gui.core.SpreadSheetTable;
import adams.gui.event.RecentFileEvent;
import adams.gui.event.RecentFileListener;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;

/**
 * A panel for viewing SpreadSheet files.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4129 $
 */
public class SpreadSheetViewerPanel
  extends BasePanel
  implements MenuBarProvider, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -7759194648757624838L;

  /** the name of the props file. */
  public final static String FILENAME = "SpreadSheetViewer.props";

  /**
   * A specialized tabbed pane with a few methods for easier access.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4129 $
   */
  public static class TabbedPane
    extends BaseTabbedPane {

    /** for serialization. */
    private static final long serialVersionUID = -2048229771213837710L;

    /**
     * Returns the table at the specified position.
     *
     * @param index	the tab index of the table
     * @return		the table
     */
    public SpreadSheetTable getTableAt(int index) {
      SpreadSheetTable	result;
      JScrollPane	pane;

      if (getComponentAt(index) instanceof JScrollPane) {
	pane   = (JScrollPane) getComponentAt(index);
	result = (SpreadSheetTable) pane.getViewport().getView();
      }
      else {
	result = (SpreadSheetTable) getComponentAt(index);
      }

      return result;
    }

    /**
     * Sets the number of decimals to use.
     *
     * @param index	the tab index
     * @param numDec	the number of decimals to use
     */
    public void setNumDecimalsAt(int index, int numDec) {
      getTableAt(index).setNumDecimals(numDec);
    }

    /**
     * returns the number of decimals in use.
     *
     * @param index	the tab index
     * @return		the number of decimals in use
     */
    public int getNumDecimalsAt(int index) {
      return getTableAt(index).getNumDecimals();
    }

    /**
     * Sets the number of decimals to use for all tables.
     *
     * @param numDec	the number of decimals to use
     */
    public void setNumDecimals(int numDec) {
      int	i;

      for (i = 0; i < getTabCount(); i++)
	setNumDecimalsAt(i, numDec);
    }
  }

  /** the file to store the recent files in. */
  public final static String SESSION_FILE = "CSVViewerSession.props";

  /** the properties. */
  protected static Properties m_Properties;

  /** the tabbed pane for displaying the CSV files. */
  protected TabbedPane m_TabbedPane;

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "open" menu item. */
  protected JMenuItem m_MenuItemFileOpen;

  /** the "load recent" submenu. */
  protected JMenu m_MenuItemFileOpenRecent;

  /** the "close" menu item. */
  protected JMenuItem m_MenuItemFileClose;

  /** the "exit" menu item. */
  protected JMenuItem m_MenuItemFileExit;

  /** the "displayed decimals" menu item. */
  protected JMenuItem m_MenuItemViewDisplayedDecimals;

  /** the "displayed decimals (all)" menu item. */
  protected JMenuItem m_MenuItemViewDisplayedDecimalsAll;

  /** the filedialog for loading CSV files. */
  protected SpreadSheetFileChooser m_FileChooser;

  /** the recent files handler. */
  protected RecentFilesHandler m_RecentFilesHandler;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_FileChooser = new SpreadSheetFileChooser();
    m_FileChooser.setMultiSelectionEnabled(true);

    m_RecentFilesHandler = null;
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    super.initGUI();

    setLayout(new BorderLayout());

    m_TabbedPane = new TabbedPane();
    m_TabbedPane.setCloseTabsWithMiddelMouseButton(true);
    add(m_TabbedPane, BorderLayout.CENTER);
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    JMenuBar	result;
    JMenu	menu;
    JMenu	submenu;
    JMenuItem	menuitem;

    if (m_MenuBar == null) {
      result = new JMenuBar();

      // File
      menu = new JMenu("File");
      result.add(menu);
      menu.setMnemonic('F');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // File/Open
      menuitem = new JMenuItem("Open...");
      menu.add(menuitem);
      menuitem.setMnemonic('O');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed O"));
      menuitem.setIcon(GUIHelper.getIcon("open.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  open();
	}
      });
      m_MenuItemFileOpen = menuitem;

      // File/Recent files
      submenu = new JMenu("Open recent");
      menu.add(submenu);
      m_RecentFilesHandler = new RecentFilesHandler(
	  SESSION_FILE, getProperties().getInteger("MaxRecentFiles", 5), submenu);
      m_RecentFilesHandler.addRecentFileListener(new RecentFileListener() {
	public void recentFileAdded(RecentFileEvent e) {
	  // ignored
	}
	public void recentFileSelected(RecentFileEvent e) {
	  load(e.getFile());
	}
      });
      m_MenuItemFileOpenRecent = submenu;

      // File/Close tab
      menuitem = new JMenuItem("Close tab");
      menu.add(menuitem);
      menuitem.setMnemonic('t');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed W"));
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  closeFile();
	}
      });
      m_MenuItemFileClose = menuitem;

      // File/Send to
      menu.addSeparator();
      if (SendToActionUtils.addSendToSubmenu(this, menu))
	menu.addSeparator();

      // File/Close
      menuitem = new JMenuItem("Close");
      menu.add(menuitem);
      menuitem.setMnemonic('C');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Q"));
      menuitem.setIcon(GUIHelper.getIcon("exit.png"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  close();
	}
      });
      m_MenuItemFileExit = menuitem;

      // View
      menu = new JMenu("View");
      result.add(menu);
      menu.setMnemonic('V');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // View/Displayed decimals
      menuitem = new JMenuItem("Displayed decimals (current file)...");
      menu.add(menuitem);
      menuitem.setMnemonic('d');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed D"));
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  String valueStr = JOptionPane.showInputDialog(
	      "Please enter the number of decimals to display (-1 to display all):",
	      m_TabbedPane.getNumDecimalsAt(m_TabbedPane.getSelectedIndex()));
	  if (valueStr == null)
	    return;
	  int decimals = Integer.parseInt(valueStr);
	  m_TabbedPane.setNumDecimalsAt(m_TabbedPane.getSelectedIndex(), decimals);
	}
      });
      m_MenuItemViewDisplayedDecimals = menuitem;

      // View/Displayed decimals
      menuitem = new JMenuItem("Displayed decimals (all open files)...");
      menu.add(menuitem);
      menuitem.setMnemonic('i');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed D"));
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  String valueStr = JOptionPane.showInputDialog(
	      "Please enter the number of decimals to display (-1 to display all):");
	  if (valueStr == null)
	    return;
	  int decimals = Integer.parseInt(valueStr);
	  m_TabbedPane.setNumDecimals(decimals);
	}
      });
      m_MenuItemViewDisplayedDecimalsAll = menuitem;

      // update menu
      m_MenuBar = result;
      updateMenu();
    }
    else {
      result = m_MenuBar;
    }

    return result;
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    if (m_MenuBar == null)
      return;

    m_MenuItemFileClose.setEnabled((m_TabbedPane.getTabCount() > 0) && (m_TabbedPane.getSelectedIndex() != -1));
  }

  /**
   * Opens one or more CSV files.
   */
  protected void open() {
    int			retVal;
    PlaceholderFile[]	files;

    retVal = m_FileChooser.showOpenDialog(this);
    if (retVal != SpreadSheetFileChooser.APPROVE_OPTION)
      return;

    files = m_FileChooser.getSelectedPlaceholderFiles();
    for (File file: files)
      load(m_FileChooser.getReader(), file);
  }

  /**
   * Loads the specified file.
   *
   * @param reader	the reader to use for reading the file
   * @param file	the file to load
   */
  public void load(File file) {
    load(m_FileChooser.getReaderForFile(file), file);
  }

  /**
   * Loads the specified file.
   *
   * @param reader	the reader to use for reading the file
   * @param file	the file to load
   */
  public void load(AbstractSpreadSheetReader reader, File file) {
    SpreadSheet	sheet;

    if (reader == null)
      reader = new CsvSpreadSheetReader();
    sheet = reader.read(file.getAbsolutePath());
    if (sheet == null) {
      GUIHelper.showErrorMessage(
	  this, "Error loading spreadsheet file:\n" + file);
    }
    else {
      m_TabbedPane.addTab(file.getName(), new BaseScrollPane(new SpreadSheetTable(sheet)));
      m_TabbedPane.setSelectedIndex(m_TabbedPane.getTabCount() - 1);

      m_FileChooser.setCurrentDirectory(file.getParentFile());
      if (m_RecentFilesHandler != null)
	m_RecentFilesHandler.addRecentFile(file);
    }
  }

  /**
   * Closes the current active tab.
   */
  protected void closeFile() {
    int		index;

    index = m_TabbedPane.getSelectedIndex();
    if (index == -1)
      return;

    m_TabbedPane.remove(index);
  }

  /**
   * Closes the dialog or frame.
   */
  protected void close() {
    closeParent();
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    return new Class[]{PlaceholderFile.class, JTable.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the classes to retrieve the item for
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    return    (SendToActionUtils.isAvailable(new Class[]{PlaceholderFile.class, JTable.class}, cls))
           && (m_TabbedPane.getSelectedIndex() != -1);
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    Object			result;
    CsvSpreadSheetWriter	writer;
    SpreadSheet			sheet;
    int				index;

    index = m_TabbedPane.getSelectedIndex();
    if (index == -1)
      return null;

    result = null;

    if (SendToActionUtils.isAvailable(PlaceholderFile.class, cls)) {
      sheet  = m_TabbedPane.getTableAt(index).getSheet();
      result = SendToActionUtils.nextTmpFile("spreadsheetviewer", "csv");
      writer = new CsvSpreadSheetWriter();
      if (!writer.write(sheet, (PlaceholderFile) result))
	result = null;
    }
    else if (SendToActionUtils.isAvailable(JTable.class, cls)) {
      result = m_TabbedPane.getTableAt(index);
    }

    return result;
  }

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(SpreadSheetViewerPanelDefinition.KEY);

    return m_Properties;
  }
}
