/*
 * SpreadSheetFileChooser.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.util.Collections;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import adams.core.io.AbstractSpreadSheetReader;
import adams.core.io.AbstractSpreadSheetWriter;
import adams.core.io.CsvSpreadSheetReader;
import adams.core.io.CsvSpreadSheetWriter;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.goe.GenericObjectEditor;
import adams.gui.goe.GenericObjectEditorDialog;

/**
 * A specialized JFileChooser that lists all available file Readers and Writers
 * for spreadsheets.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4129 $
 */
public class SpreadSheetFileChooser
  extends AbstractExtensionFileFilterFileChooser<SpreadSheetFileChooser.SpreadSheetFileExtensionFilter>
  implements FileTypeDeterminingFileChooser<AbstractSpreadSheetReader,AbstractSpreadSheetWriter> {

  /** for serialization. */
  private static final long serialVersionUID = -6341967475735162796L;

  /**
   * A custom filter class that stores the associated class along the
   * description and extensions.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4129 $
   */
  public static class SpreadSheetFileExtensionFilter
    extends ExtensionFileFilter {

    /** for serialization. */
    private static final long serialVersionUID = -2968757331524482453L;

    /** the classname. */
    protected String m_Classname;

    /**
     * Constructs a filter that matches all files.
     *
     * @param classname		the classname this filter is for
     */
    public SpreadSheetFileExtensionFilter(String classname) {
      super();

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     */
    public SpreadSheetFileExtensionFilter(String classname, String description, String extension) {
      super(description, extension);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     */
    public SpreadSheetFileExtensionFilter(String classname, String description, String[] extensions) {
      super(description, extensions);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public SpreadSheetFileExtensionFilter(String classname, String description, String extension, boolean caseSensitive) {
      super(description, extension, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public SpreadSheetFileExtensionFilter(String classname, String description, String[] extensions, boolean caseSensitive) {
      super(description, extensions, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Returns the associated classname.
     *
     * @return		the classname
     */
    public String getClassname() {
      return m_Classname;
    }
  }

  /** the file filters for the readers. */
  protected static Vector<SpreadSheetFileExtensionFilter> m_ReaderFileFilters;

  /** the file filters for the writers. */
  protected static Vector<SpreadSheetFileExtensionFilter> m_WriterFileFilters;

  /** the configure button. */
  protected JButton m_ConfigureButton;

  /** the checkbox for bringing up the GenericObjectEditor. */
  protected JCheckBox m_CheckBoxOptions;

  /** the note about the options dialog. */
  protected JLabel m_LabelOptions;

  /** the GOE for displaying the options of a reader/writer. */
  protected GenericObjectEditor m_Editor;

  /**
   * Constructs a FileChooser pointing to the user's default directory.
   */
  public SpreadSheetFileChooser() {
    super();
  }

  /**
   * Constructs a FileChooser using the given File as the path.
   *
   * @param currentDirectory	the path to start in
   */
  public SpreadSheetFileChooser(File currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Constructs a FileChooser using the given path.
   *
   * @param currentDirectory	the path to start in
   */
  public SpreadSheetFileChooser(String currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Further initializations.
   */
  protected void initialize() {
    JPanel	panel;
    JPanel	panel2;

    super.initialize();

    m_CheckBoxOptions = new JCheckBox("Invoke options dialog");
    m_CheckBoxOptions.setMnemonic('I');
    m_LabelOptions = new JLabel("<html><br>Note:<br><br>Some file formats offer additional<br>options which can be customized<br>when invoking the options dialog.</html>");
    panel = new JPanel(new BorderLayout());
    panel.add(m_CheckBoxOptions, BorderLayout.NORTH);
    panel2 = new JPanel(new BorderLayout());
    panel2.add(m_LabelOptions, BorderLayout.NORTH);
    panel2.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panel.add(panel2, BorderLayout.CENTER);
    setAccessory(panel);

    m_Editor = new GenericObjectEditor(false);

    setAutoAppendExtension(true);
  }

  /**
   * Returns whether the filters have already been initialized.
   *
   * @return		true if the filters have been initialized
   */
  protected boolean getFiltersInitialized() {
    return (m_ReaderFileFilters != null);
  }

  /**
   * Performs the actual initialization of the filters.
   */
  protected void doInitializeFilters() {
    initFilters(true, AbstractSpreadSheetReader.getReaders());
    initFilters(false, AbstractSpreadSheetWriter.getWriters());
  }

  /**
   * initializes the SpreadSheetFileExtensionFilters.
   *
   * @param reader	if true then the reader filters are initialized
   * @param classnames	the classnames of the converters
   */
  protected static void initFilters(boolean reader, String[] classnames) {
    int					i;
    String 				classname;
    Class 				cls;
    String[] 				ext;
    String 				desc;
    Object		 		converter;
    SpreadSheetFileExtensionFilter 	filter;

    if (reader)
      m_ReaderFileFilters = new Vector<SpreadSheetFileExtensionFilter>();
    else
      m_WriterFileFilters  = new Vector<SpreadSheetFileExtensionFilter>();

    for (i = 0; i < classnames.length; i++) {
      classname = (String) classnames[i];

      // get data from converter
      try {
	cls       = Class.forName(classname);
	converter = cls.newInstance();
	if (reader) {
	  desc = ((AbstractSpreadSheetReader) converter).getFormatDescription();
	  ext  = ((AbstractSpreadSheetReader) converter).getFormatExtensions();
	}
	else {
	  desc = ((AbstractSpreadSheetWriter) converter).getFormatDescription();
	  ext  = ((AbstractSpreadSheetWriter) converter).getFormatExtensions();
	}
      }
      catch (Exception e) {
	cls       = null;
	converter = null;
	ext       = new String[0];
	desc      = "";
      }

      if (converter == null)
	continue;

      // reader?
      if (reader) {
	filter = new SpreadSheetFileExtensionFilter(classname, desc, ext);
	m_ReaderFileFilters.add(filter);
      }
      else {
	filter = new SpreadSheetFileExtensionFilter(classname, desc, ext);
	m_WriterFileFilters.add(filter);
      }
    }

    if (reader)
      Collections.sort(m_ReaderFileFilters);
    else
      Collections.sort(m_WriterFileFilters);
  }

  /**
   * Returns the file filters for opening files.
   *
   * @return		the file filters
   */
  protected Vector<SpreadSheetFileExtensionFilter> getOpenFileFilters() {
    return m_ReaderFileFilters;
  }

  /**
   * Returns the file filters for writing files.
   *
   * @return		the file filters
   */
  protected Vector<SpreadSheetFileExtensionFilter> getSaveFileFilters() {
    return m_WriterFileFilters;
  }

  /**
   * initializes the GUI.
   *
   * @param dialogType		the type of dialog to setup the GUI for
   */
  protected void initGUI(int dialogType) {
    super.initGUI(dialogType);

    // initial setup
    if (dialogType == OPEN_DIALOG) {
      m_Editor.setClassType(AbstractSpreadSheetReader.class);
      m_Editor.setValue(getDefaultReader());
    }
    else {
      m_Editor.setClassType(AbstractSpreadSheetWriter.class);
      m_Editor.setValue(getDefaultWriter());
    }
    restoreLastFilter(dialogType);
  }

  /**
   * Returns the default reader.
   *
   * @return		the default reader
   */
  protected AbstractSpreadSheetReader getDefaultReader() {
    return new CsvSpreadSheetReader();
  }

  /**
   * Returns the default writer.
   *
   * @return		the default writer
   */
  protected AbstractSpreadSheetWriter getDefaultWriter() {
    return new CsvSpreadSheetWriter();
  }

  /**
   * Pops up an "Open File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showOpenDialog(Component parent) {
    int result = super.showOpenDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * Pops up an "Save File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showSaveDialog(Component parent) {
    int result = super.showSaveDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * returns the reader that was chosen by the user, can be null in case the
   * user aborted the dialog or the save dialog was shown.
   *
   * @return		the chosen reader, if any
   */
  public AbstractSpreadSheetReader getReader() {
    configureCurrentHandlerHook(OPEN_DIALOG);

    if (m_CurrentHandler instanceof AbstractSpreadSheetWriter)
      return null;
    else
      return (AbstractSpreadSheetReader) m_CurrentHandler;
  }

  /**
   * returns the writer that was chosen by the user, can be null in case the
   * user aborted the dialog or the open dialog was shown.
   *
   * @return		the chosen writer, if any
   */
  public AbstractSpreadSheetWriter getWriter() {
    configureCurrentHandlerHook(SAVE_DIALOG);

    if (m_CurrentHandler instanceof AbstractSpreadSheetReader)
      return null;
    else
      return (AbstractSpreadSheetWriter) m_CurrentHandler;
  }

  /**
   * sets the current converter according to the current filefilter.
   */
  protected void updateCurrentHandlerHook() {
    String	classname;
    Object	newHandler;

    try {
      // determine current converter
      classname  = ((SpreadSheetFileExtensionFilter) getFileFilter()).getClassname();
      newHandler = Class.forName(classname).newInstance();

      if (m_CurrentHandler == null) {
	m_CurrentHandler = newHandler;
      }
      else {
	if (!m_CurrentHandler.getClass().equals(newHandler.getClass()))
	  m_CurrentHandler = newHandler;
      }
      setFileSelectionMode(FILES_ONLY);
    }
    catch (Exception e) {
      m_CurrentHandler = null;
      e.printStackTrace();
    }
  }

  /**
   * configures the current converter.
   *
   * @param dialogType		the type of dialog to configure for
   */
  protected void configureCurrentHandlerHook(int dialogType) {
    String		classname;

    if (m_CurrentHandler == null) {
      classname = ((SpreadSheetFileExtensionFilter) getFileFilter()).getClassname();
      try {
	m_CurrentHandler = Class.forName(classname).newInstance();
      }
      catch (Exception e) {
	e.printStackTrace();
	m_CurrentHandler = null;
      }

      // none found?
      if (m_CurrentHandler == null)
	return;
    }
  }

  /**
   * Returns the reader for the specified file.
   *
   * @param file	the file to determine a reader for
   * @return		the reader, null if none found
   */
  public AbstractSpreadSheetReader getReaderForFile(File file) {
    AbstractSpreadSheetReader	result;

    result = null;

    for (SpreadSheetFileExtensionFilter filter: m_ReaderFileFilters) {
      if (filter.accept(file)) {
	try {
	  result = (AbstractSpreadSheetReader) Class.forName(filter.getClassname()).newInstance();
	}
	catch (Exception e) {
	  System.err.println("Failed to instantiate reader '" + filter.getClassname() + "':");
	  e.printStackTrace();
	}
      }
    }

    return result;
  }

  /**
   * Returns the writer for the specified file.
   *
   * @param file	the file to determine a reader for
   * @return		the writer, null if none found
   */
  public AbstractSpreadSheetWriter getWriterForFile(File file) {
    AbstractSpreadSheetWriter	result;

    result = null;

    for (SpreadSheetFileExtensionFilter filter: m_WriterFileFilters) {
      if (filter.accept(file)) {
	try {
	  result = (AbstractSpreadSheetWriter) Class.forName(filter.getClassname()).newInstance();
	}
	catch (Exception e) {
	  System.err.println("Failed to instantiate writer '" + filter.getClassname() + "':");
	  e.printStackTrace();
	}
      }
    }

    return result;
  }
}
