/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetSubsetTest.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import junit.framework.Test;
import junit.framework.TestSuite;
import adams.env.Environment;
import adams.flow.AbstractFlowTest;
import adams.flow.control.Branch;
import adams.flow.control.Flow;
import adams.flow.control.Sequence;
import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorReference;
import adams.flow.standalone.GlobalActors;
import adams.flow.sink.DumpFile;
import adams.flow.sink.GlobalSink;
import adams.flow.source.SingleFileSupplier;
import adams.test.TmpFile;

/**
 * Tests the SpreadSheetSubset actor.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class SpreadSheetSubsetTest
  extends AbstractFlowTest {

  /**
   * Initializes the test.
   *
   * @param name	the name of the test
   */
  public SpreadSheetSubsetTest(String name) {
    super(name);
  }

  /**
   * Called by JUnit before each test method. Copies some resource files into
   * the tmp directory
   *
   * @throws Exception if an error occurs reading the example instances.
   */
  protected void setUp() throws Exception {
    super.setUp();

    m_TestHelper.copyResourceToTmp("iris.csv");
  }

  /**
   * Called by JUnit after each test method. Removes some tmp files again.
   *
   * @throws Exception	if tear-down fails
   */
  protected void tearDown() throws Exception {
    m_TestHelper.deleteFileFromTmp("iris.csv");
    m_TestHelper.deleteFileFromTmp("dumpfile.txt");

    super.tearDown();
  }

  /**
   * Used to create an instance of a specific actor.
   *
   * @return a suitably configured <code>AbstractActor</code> value
   */
  public AbstractActor getActor() {
    DumpFile df = new DumpFile();
    df.setName("out");
    df.setAppend(true);
    df.setOutputFile(new TmpFile("dumpfile.txt"));

    GlobalActors ga = new GlobalActors();
    ga.setActors(new AbstractActor[]{
	df
    });

    SingleFileSupplier sfs = new SingleFileSupplier();
    sfs.setFile(new TmpFile("iris.csv"));

    SpreadSheetReader ssr = new SpreadSheetReader();

    SpreadSheetSubset ssc1 = new SpreadSheetSubset();
    ssc1.setRows("10-20");
    ssc1.setColumns("1,3,last");

    GlobalSink gs1 = new GlobalSink();
    gs1.setGlobalName(new GlobalActorReference("out"));

    Sequence seq1 = new Sequence();
    seq1.setActors(new AbstractActor[]{
	ssc1,
	gs1
    });

    SpreadSheetSubset ssc2 = new SpreadSheetSubset();
    ssc2.setRows("20-30");
    ssc2.setColumns("2,4,last");

    GlobalSink gs2 = new GlobalSink();
    gs2.setGlobalName(new GlobalActorReference("out"));

    Sequence seq2 = new Sequence();
    seq2.setActors(new AbstractActor[]{
	ssc2,
	gs2
    });

    SpreadSheetSubset ssc3 = new SpreadSheetSubset();
    ssc3.setRows("30-40");
    ssc3.setColumns("1,2,last");

    GlobalSink gs3 = new GlobalSink();
    gs3.setGlobalName(new GlobalActorReference("out"));

    Sequence seq3 = new Sequence();
    seq3.setActors(new AbstractActor[]{
	ssc3,
	gs3
    });

    Branch br = new Branch();
    br.setNumThreads(0);
    br.setBranches(new AbstractActor[]{
	seq1,
	seq2,
	seq3
    });

    Flow flow = new Flow();
    flow.setActors(new AbstractActor[]{ga, sfs, ssr, br});

    return flow;
  }

  /**
   * Performs a regression test, comparing against previously generated output.
   */
  public void testRegression() {
    performRegressionTest(
	new TmpFile("dumpfile.txt"));
  }

  /**
   * Returns a test suite.
   *
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(SpreadSheetSubsetTest.class);
  }

  /**
   * Runs the test from commandline.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    runTest(suite());
  }
}
