/*
 * SpreadSheetColumnsByName.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Vector;

import adams.core.base.BaseRegExp;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Creates a new spreadsheet with the columns that matched the regular expression. It is possible to invert the matching sense as well.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetColumnsByName
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-reg-exp &lt;adams.core.base.BaseRegExp&gt; (property: regExp)
 * &nbsp;&nbsp;&nbsp;The regular expression to match the column names against.
 * &nbsp;&nbsp;&nbsp;default: .*
 * </pre>
 *
 * <pre>-invert-matching (property: invertMatching)
 * &nbsp;&nbsp;&nbsp;If enabled, the matching sense is inverted, ie, only non-matching columns
 * &nbsp;&nbsp;&nbsp;get output.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3509 $
 */
public class SpreadSheetColumnsByName
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -253714973019682939L;

  /** the regular expression to match the column names against. */
  protected BaseRegExp m_RegExp;

  /** whether to invert the matching. */
  protected boolean m_InvertMatching;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Creates a new spreadsheet with the columns that matched the regular "
     + "expression. It is possible to invert the matching sense as well.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "reg-exp", "regExp",
	    new BaseRegExp());

    m_OptionManager.add(
	    "invert-matching", "invertMatching",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "";
    if (m_InvertMatching)
      result += "not ";
    result += "matching: ";

    variable = getOptionManager().getVariableForProperty("columns");
    if (variable != null)
      result += variable;
    else
      result += m_RegExp;

    return result;
  }

  /**
   * Sets the regular expression to use for matching column names.
   *
   * @param value	the regular expression
   */
  public void setRegExp(BaseRegExp value) {
    m_RegExp = value;
    reset();
  }

  /**
   * Returns the regular expression to use for matching column names.
   *
   * @return		the regular expression
   */
  public BaseRegExp getRegExp() {
    return m_RegExp;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String regExpTipText() {
    return "The regular expression to match the column names against.";
  }

  /**
   * Sets the whether to invert the matching.
   *
   * @param value	if true the matching is inverted
   */
  public void setInvertMatching(boolean value) {
    m_InvertMatching = value;
    reset();
  }

  /**
   * Returns whether to invert the matching.
   *
   * @return		true if the matching is inverted
   */
  public boolean getInvertMatching() {
    return m_InvertMatching;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String invertMatchingTipText() {
    return "If enabled, the matching sense is inverted, ie, only non-matching columns get output.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.core.io.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.core.io.SpreadSheet.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    SpreadSheet		sheet;
    SpreadSheet		subset;
    Row			row;
    Row			subrow;
    Vector<Integer>	indices;
    int			i;
    int			n;
    boolean		add;

    result = null;

    sheet = (SpreadSheet) m_InputToken.getPayload();

    // determine columns for output
    indices = new Vector<Integer>();
    for (i = 0; i < sheet.getColumnCount(); i++) {
      if (m_InvertMatching)
	add = !m_RegExp.isMatch(sheet.getHeaderRow().getCell(i).getContent());
      else
	add = m_RegExp.isMatch(sheet.getHeaderRow().getCell(i).getContent());
      if (add)
	indices.add(i);
    }

    if (indices.size() > 0) {
      if (isDebugOn())
	debug("Output columns: " + indices);

      subset = new SpreadSheet();
      // header
      for (i = 0; i < indices.size(); i++) {
	subset.getHeaderRow().addCell("" + (i+1)).setContent(
	    sheet.getHeaderRow().getCell(indices.get(i)).getContent());
      }
      // data
      for (n = 0; n < sheet.getRowCount(); n++) {
	row    = sheet.getRow(n);
	subrow = subset.addRow("" + (subset.getRowCount()));
	for (i = 0; i < indices.size(); i++) {
	  subrow.addCell("" + (i+1)).setContent(
	      row.getCell(indices.get(i)).getContent());
	}
      }

      m_OutputToken = new Token(subset);
    }
    else {
      if (isDebugOn())
	debug("No columns for output!");
    }

    return result;
  }
}
