/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * TransposeSpreadSheet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.data.conversion;

import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;

/**
 <!-- globalinfo-start -->
 * Converts adams.data.instance.Instance objects into weka.core.Instance ones.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class TransposeSpreadSheet
  extends AbstractConversion {

  /** for serialization. */
  private static final long serialVersionUID = 4733940137387662202L;

  /** whether to exclude header and only transpose data. */
  protected boolean m_OnlyData;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Transposes a spreadsheet, i.e., swaps columns with rows.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "only-data", "onlyData",
	    false);
  }

  /**
   * Sets whether to transpose only the data, excluding the header.
   *
   * @param value	true if to exclude the header
   */
  public void setOnlyData(boolean value) {
    m_OnlyData = value;
    reset();
  }

  /**
   * Returns whether to transpose only the data, excluding the header.
   *
   * @return		true if the header is excluded
   */
  public boolean getOnlyData() {
    return m_OnlyData;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String onlyDataTipText() {
    return "If enabled, only the data gets transposed and not the header row as well.";
  }

  /**
   * Returns the class that is accepted as input.
   *
   * @return		the class
   */
  public Class accepts() {
    return SpreadSheet.class;
  }

  /**
   * Returns the class that is generated as output.
   *
   * @return		the class
   */
  public Class generates() {
    return SpreadSheet.class;
  }

  /**
   * Performs the actual conversion.
   *
   * @return		the converted data
   * @throws Exception	if something goes wrong with the conversion
   */
  protected Object doConvert() throws Exception {
    SpreadSheet		result;
    SpreadSheet		input;
    int			i;
    int			n;
    Row			rowIn;
    Row			rowOut;
    String		value;

    result = new SpreadSheet();
    input  = (SpreadSheet) getInput();

    // header
    rowOut = result.getHeaderRow();
    for (i = 0; i < input.getRowCount(); i++) {
      if (!m_OnlyData && (i == 0))
	rowOut.addCell("Header").setContent("Header");
      rowOut.addCell("Row-" + (i+1)).setContent("Row-" + (i+1));
    }
    // create empty sheet
    for (n = 0; n < input.getColumnCount(); n++) {
      rowOut = result.addRow("Col-" + (result.getRowCount()+1));
      for (i = 0; i < input.getRowCount(); i++) {
	if (!m_OnlyData && (i == 0))
	  rowOut.addCell("Header").setContent("Header");
	rowOut.addCell("Row-" + (i+1));
      }
    }
    // transpose header
    if (!m_OnlyData) {
      rowIn = input.getHeaderRow();
      for (n = 0; n < rowIn.getCellCount(); n++) {
	value = rowIn.getCell(n).getContent();
	rowOut = result.getRow(n);
	rowOut.getCell("Header").setContent(value);
      }
    }
    // transpose data
    for (i = 0; i < input.getRowCount(); i++) {
      rowIn = input.getRow(i);
      for (n = 0; n < rowIn.getCellCount(); n++) {
	value = rowIn.getCell(n).getContent();
	rowOut = result.getRow(n);
	rowOut.getCell("Row-" + (i+1)).setContent(value);
      }
    }

    return result;
  }
}
