/*
 * SpreadSheetPlotGenerator.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Collections;
import java.util.Vector;

import adams.core.Index;
import adams.core.Range;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Cell;
import adams.core.io.SpreadSheet.Row;
import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Outputs plot containers generated from a spreadsheet.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetPlotGenerator
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-columns &lt;java.lang.String&gt; (property: plotColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to include in the plot; A range is a comma-separated
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); the
 * &nbsp;&nbsp;&nbsp;following placeholders can be used as well: first, second, third, last_2,
 * &nbsp;&nbsp;&nbsp; last_1, last
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-sort-column &lt;java.lang.String&gt; (property: sortColumn)
 * &nbsp;&nbsp;&nbsp;The (optional) index of the column to use for sorting; An index is a number
 * &nbsp;&nbsp;&nbsp;starting with 1; the following placeholders can be used as well: first,
 * &nbsp;&nbsp;&nbsp;second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-sort-descending (property: sortDescending)
 * &nbsp;&nbsp;&nbsp;If set to true, sorting is done in descending manner instead (ignored is
 * &nbsp;&nbsp;&nbsp;using a column for X values).
 * </pre>
 *
 * <pre>-x-column &lt;java.lang.String&gt; (property: XColumn)
 * &nbsp;&nbsp;&nbsp;The (optional) index of the column which values to use as X values in the
 * &nbsp;&nbsp;&nbsp;plot; An index is a number starting with 1; the following placeholders can
 * &nbsp;&nbsp;&nbsp;be used as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-default-cell-value &lt;double&gt; (property: defaultCellValue)
 * &nbsp;&nbsp;&nbsp;The default value for missing or non-numeric cells.
 * &nbsp;&nbsp;&nbsp;default: -1.0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2415 $
 */
public class SpreadSheetPlotGenerator
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 1147935218531182101L;

  /**
   * Helper class for sorting rows.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 2415 $
   */
  public static class SortableRow
    implements Comparable<SortableRow> {

    /** the index. */
    protected Comparable m_Index;

    /** the row. */
    protected Row m_Row;

    /** whether to invert the sorting order. */
    protected boolean m_InvertSorting;

    /**
     * Initializes the container.
     *
     * @param index	the index used for sorting
     * @param row	the row payload
     * @param invSort	whether to invert sorting
     */
    public SortableRow(Comparable index, Row row, boolean invSort) {
      super();

      m_Index         = index;
      m_Row           = row;
      m_InvertSorting = invSort;
    }

    /**
     * Returns the sort index.
     *
     * @return		the index
     */
    public Comparable getIndex() {
      return m_Index;
    }

    /**
     * Returns the row payload.
     *
     * @return		the row
     */
    public Row getRow() {
      return m_Row;
    }

    /**
     * Returns whether sorting is inverted.
     *
     * @return		true if sorting is inverted
     */
    public boolean isInvertSorting() {
      return m_InvertSorting;
    }

    /**
     * Compares this object with the specified object for order.  Returns a
     * negative integer, zero, or a positive integer as this object is less
     * than, equal to, or greater than the specified object.
     *
     * Takes the inverted sorting into account.
     *
     * @param   o the object to be compared.
     * @return  a negative integer, zero, or a positive integer as this object
     *		is less than, equal to, or greater than the specified object.
     *
     * @throws ClassCastException if the specified object's type prevents it
     *         from being compared to this object.
     * @see    #isInvertSorting()
     */
    public int compareTo(SortableRow o) {
      return getIndex().compareTo(o.getIndex()) * (m_InvertSorting ? -1 : 1);
    }

    /**
     * Returns a short string representation of the container.
     *
     * @return		the string representation
     */
    public String toString() {
      return (m_InvertSorting ? "!" : "") + m_Index + ": " + m_Row;
    }
  }

  /** the default string value for cells that are missing. */
  public final static String MISSING_CELL_VALUE = "MISSING";

  /** the columns to plot. */
  protected Range m_PlotColumns;

  /** the (optional) column to sort on. */
  protected Index m_SortColumn;

  /** whether to sort descending (if sorting). */
  protected boolean m_SortDescending;

  /** the (optional) column to use as X value. */
  protected Index m_XColumn;

  /** the default value for missing or non-numeric cells. */
  protected double m_DefaultCellValue;

  /** the generated plot containers. */
  protected Vector<SequencePlotterContainer> m_Containers;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Outputs plot containers generated from a spreadsheet.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "plotColumns",
	    "");

    m_OptionManager.add(
	    "sort-column", "sortColumn",
	    "");

    m_OptionManager.add(
	    "sort-descending", "sortDescending",
	    false);

    m_OptionManager.add(
	    "x-column", "XColumn",
	    "");

    m_OptionManager.add(
	    "default-cell-value", "defaultCellValue",
	    -1.0);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Containers  = new Vector<SequencePlotterContainer>();
    m_PlotColumns = new Range();
    m_SortColumn  = new Index();
    m_XColumn     = new Index();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("plotColumns");
    if (variable != null)
      return variable;
    else if (m_PlotColumns.getRange().length() > 0)
      return m_PlotColumns.getRange();
    else
      return null;
  }

  /**
   * Sets the column range to use in the plot.
   *
   * @param value	the column range
   */
  public void setPlotColumns(String value) {
    m_PlotColumns.setRange(value);
    reset();
  }

  /**
   * Returns the current column range to use in the plot.
   *
   * @return		the column range
   */
  public String getPlotColumns() {
    return m_PlotColumns.getRange();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String plotColumnsTipText() {
    return "The range of columns to include in the plot; " + m_PlotColumns.getExample();
  }

  /**
   * Sets the column index to use for sorting.
   *
   * @param value	the column index
   */
  public void setSortColumn(String value) {
    m_SortColumn.setIndex(value);
    reset();
  }

  /**
   * Returns the column index to use for sorting.
   *
   * @return		the column index
   */
  public String getSortColumn() {
    return m_SortColumn.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String sortColumnTipText() {
    return "The (optional) index of the column to use for sorting; " + m_SortColumn.getExample();
  }

  /**
   * Sets whether to sort in descending manner instead.
   *
   * @param value	true if to sort descendingly
   */
  public void setSortDescending(boolean value) {
    m_SortDescending = value;
    reset();
  }

  /**
   * Returns whether to sorting in descending manner instead.
   *
   * @return		true if sorting descendingly
   */
  public boolean getSortDescending() {
    return m_SortDescending;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String sortDescendingTipText() {
    return "If set to true, sorting is done in descending manner instead (ignored is using a column for X values).";
  }

  /**
   * Sets the index of the column which values to use as X values.
   *
   * @param value	the column index
   */
  public void setXColumn(String value) {
    m_XColumn.setIndex(value);
    reset();
  }

  /**
   * Returns the index of the column which values to use as X values.
   *
   * @return		the column index
   */
  public String getXColumn() {
    return m_XColumn.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String XColumnTipText() {
    return "The (optional) index of the column which values to use as X values in the plot; " + m_XColumn.getExample();
  }

  /**
   * Sets the default value for missing or non-numeric cells.
   *
   * @param value	the default value
   */
  public void setDefaultCellValue(double value) {
    m_DefaultCellValue = value;
    reset();
  }

  /**
   * Returns the default value for missing or non-numeric cells.
   *
   * @return		the default value
   */
  public double getDefaultCellValue() {
    return m_DefaultCellValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String defaultCellValueTipText() {
    return "The default value for missing or non-numeric cells.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.core.io.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the string value for the specified cell.
   *
   * @param row		the row to get the cell from
   * @param index	the column index
   * @return		the string value
   */
  protected String getCellString(Row row, int index) {
    String	result;
    Cell	cell;

    result = MISSING_CELL_VALUE;

    cell = row.getCell(index);
    if ((cell != null) && !cell.isMissing())
      result = cell.getContent();

    return result;
  }

  /**
   * Returns the cell value for the specified column index.
   * Uses the default value if the cell is missing or non-numeric.
   *
   * @param row		the row to get the cell from
   * @param index	the column index
   * @return		the cell value
   * @see		#m_DefaultCellValue
   */
  protected double getCellValue(Row row, int index) {
    double	result;
    Cell	cell;

    result = m_DefaultCellValue;

    cell = row.getCell(index);
    if ((cell != null) && !cell.isMissing() && cell.isNumeric())
      result = Double.parseDouble(cell.getContent());

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    SequencePlotterContainer	cont;
    SpreadSheet			sheet;
    int				i;
    int				n;
    int				sortIndex;
    boolean			sortNumeric;
    Row				row;
    Vector<SortableRow>		srows;
    SortableRow			srow;
    int[]			plotCols;
    String[]			plotNames;
    int				xIndex;

    result = null;

    sheet = (SpreadSheet) m_InputToken.getPayload();
    m_PlotColumns.setMax(sheet.getColumnCount());
    m_SortColumn.setMax(sheet.getColumnCount());
    m_XColumn.setMax(sheet.getColumnCount());

    // gather data
    xIndex = m_XColumn.getIntIndex();
    if ((xIndex != -1) && !sheet.isNumeric(xIndex))
      xIndex = -1;
    sortIndex   = m_SortColumn.getIntIndex();
    sortNumeric = false;
    if (sortIndex != -1)
      sortNumeric = sheet.isNumeric(sortIndex);
    srows = new Vector<SortableRow>();
    for (i = 0; i < sheet.getRowCount(); i++) {
      row = sheet.getRow(i);
      if (xIndex != -1) {
	srow = new SortableRow(new Double(getCellValue(row, xIndex)), row, false);
      }
      else {
	if (sortIndex == -1) {
	  srow = new SortableRow(new Integer(i), row, m_SortDescending);
	}
	else {
	  if (sortNumeric)
	    srow = new SortableRow(new Double(getCellValue(row, sortIndex)), row, m_SortDescending);
	  else
	    srow = new SortableRow(getCellString(row, sortIndex), row, m_SortDescending);
	}
      }
      srows.add(srow);
    }
    Collections.sort(srows);

    // create plot containers
    m_Containers.clear();
    plotCols  = m_PlotColumns.getIntIndices();
    plotNames = new String[plotCols.length];
    for (i = 0; i < plotNames.length; i++)
      plotNames[i] = sheet.getHeaderRow().getCell(plotCols[i]).getContent();
    for (i = 0; i < srows.size(); i++) {
      row = srows.get(i).getRow();
      for (n = 0; n < plotCols.length; n++) {
	if (xIndex == -1)
	  cont = new SequencePlotterContainer(plotNames[n], new Double(i), getCellValue(row, plotCols[n]));
	else
	  cont = new SequencePlotterContainer(plotNames[n], getCellValue(row, xIndex), getCellValue(row, plotCols[n]));
	m_Containers.add(cont);
      }
    }

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.container.SequencePlotterContainer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{SequencePlotterContainer.class};
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_Containers.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = new Token(m_Containers.get(0));
    m_InputToken  = null;
    m_Containers.remove(0);

    return result;
  }
}
