/*
  *    SVGWriter.java
  *    Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
  *
  */

package adams.gui.print;

import java.io.BufferedWriter;
import java.io.FileWriter;

import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGraphics2D;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;

import adams.core.annotation.MixedCopyright;

/**
 <!-- globalinfo-start -->
 * Outputs scalable vector graphics (SVG).
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-file &lt;adams.core.io.PlaceholderFile&gt; (property: file)
 * &nbsp;&nbsp;&nbsp;The file to save the image to.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-custom-dimensions (property: useCustomDimensions)
 * &nbsp;&nbsp;&nbsp;Whether to use custom dimensions or use the component's ones.
 * </pre>
 *
 * <pre>-custom-width &lt;int&gt; (property: customWidth)
 * &nbsp;&nbsp;&nbsp;The custom width.
 * &nbsp;&nbsp;&nbsp;default: -1
 * </pre>
 *
 * <pre>-custom-height &lt;int&gt; (property: customHeight)
 * &nbsp;&nbsp;&nbsp;The custom height.
 * &nbsp;&nbsp;&nbsp;default: -1
 * </pre>
 *
 * <pre>-scaling (property: scalingEnabled)
 * &nbsp;&nbsp;&nbsp;If set to true, then scaling will be used.
 * </pre>
 *
 * <pre>-scale-x &lt;double&gt; (property: XScale)
 * &nbsp;&nbsp;&nbsp;The scaling factor for the X-axis.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 * <pre>-scale-y &lt;double&gt; (property: YScale)
 * &nbsp;&nbsp;&nbsp;The scaling factor for the Y axis.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 <!-- options-end -->
 * <p/>
 * Based on weka.gui.visualize.SVGWriter
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2618 $
 */
@MixedCopyright
public class SVGWriter
  extends ScalableComponentWriter {

  /** for serialization. */
  private static final long serialVersionUID = 6049112417449275816L;

  /** the SVG namespace. */
  public final static String SVG_NAMESPACE = "http://www.w3.org/2000/svg";

  /**
   * Returns a string describing the object.
   *
   * @return 		a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Outputs scalable vector graphics (SVG).";
  }

  /**
   * returns the name of the writer, to display in the FileChooser.
   *
   * @return 		the name of the writer
   */
  public String getDescription() {
    return "SVG graphic";
  }

  /**
   * returns the extensions (incl. ".") of the output format, to use in the
   * FileChooser.
   *
   * @return 		the file extensions
   */
  public String[] getExtensions() {
    return new String[]{".svg"};
  }

  /**
   * generates the actual output.
   *
   * @throws Exception	if something goes wrong
   */
  public void generateOutput() throws Exception {
    DOMImplementation 	domImpl;
    Document 		document;
    boolean 		useCSS;
    SVGGraphics2D 	svgGenerator;
    BufferedWriter	writer;

    // Get a DOMImplementation.
    domImpl = GenericDOMImplementation.getDOMImplementation();

    // Create an instance of org.w3c.dom.Document.
    document = domImpl.createDocument(SVG_NAMESPACE, "svg", null);

    // Create an instance of the SVG Generator.
    svgGenerator = new SVGGraphics2D(document);
    svgGenerator.scale(getXScale(), getYScale());
    m_Component.paintAll(svgGenerator);

    // Finally, stream out SVG to the standard output using
    // UTF-8 encoding.
    useCSS = true; // we want to use CSS style attributes
    writer = new BufferedWriter(new FileWriter(m_OutputFile.getAbsolutePath()));
    svgGenerator.stream(writer, useCSS);
  }
}
