/*
 * TwitterConnection.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.standalone;

import twitter4j.TwitterFactory;
import twitter4j.TwitterStreamFactory;
import adams.core.base.BasePassword;

/**
 <!-- globalinfo-start -->
 * Outputs tweet messages.<br/>
 * For more information on twitter queries, see the following web page:<br/>
 * http://search.twitter.com/operators
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: TwitterConnection
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-user &lt;java.lang.String&gt; (property: user)
 * &nbsp;&nbsp;&nbsp;The twitter user to use for connecting; leave empty for public searches.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-password &lt;java.lang.String&gt; (property: password)
 * &nbsp;&nbsp;&nbsp;The password of the twitter user to use for connecting.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-query &lt;java.lang.String&gt; (property: query)
 * &nbsp;&nbsp;&nbsp;The query for obtaining the tweets.
 * &nbsp;&nbsp;&nbsp;default: search term
 * </pre>
 *
 * <pre>-results-per-page &lt;int&gt; (property: resultsPerPage)
 * &nbsp;&nbsp;&nbsp;The number of results per page when querying twitter.
 * &nbsp;&nbsp;&nbsp;default: 20
 * &nbsp;&nbsp;&nbsp;minimum: 10
 * &nbsp;&nbsp;&nbsp;maximum: 100
 * </pre>
 *
 * <pre>-max-tweets &lt;int&gt; (property: maxTweets)
 * &nbsp;&nbsp;&nbsp;The maximum number of tweets to output.
 * &nbsp;&nbsp;&nbsp;default: 100
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3548 $
 */
public class TwitterConnection
  extends AbstractStandalone {

  /** for serialization. */
  private static final long serialVersionUID = -1959430342987913960L;

  /** the twitter user to use. */
  protected String m_User;

  /** the twitter password to use. */
  protected BasePassword m_Password;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Provides access to various twitter services.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "user", "user",
	    "");

    m_OptionManager.add(
	    "password", "password",
	    new BasePassword(""));
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("user");
    if (variable != null) {
      result = variable;
    }
    else {
      if (m_User.length() == 0) {
	result = "anonymous";
	return result;
      }
      else {
	result = m_User;
      }
    }

    result  += "/";
    variable = getOptionManager().getVariableForProperty("password");
    if (variable != null)
      result += variable;
    else
      result += m_Password.getValue().replaceAll(".", "*");

    return result;
  }

  /**
   * Sets the twitter user to use.
   *
   * @param value	the user name
   */
  public void setUser(String value) {
    m_User = value;
  }

  /**
   * Returns the twitter user name to use.
   *
   * @return		the user name
   */
  public String getUser() {
    return m_User;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String userTipText() {
    return "The twitter user to use for connecting; leave empty for public searches.";
  }

  /**
   * Sets the twitter password to use.
   *
   * @param value	the password
   */
  public void setPassword(BasePassword value) {
    m_Password = value;
  }

  /**
   * Returns the twitter password to use.
   *
   * @return		the password
   */
  public BasePassword getPassword() {
    return m_Password;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String passwordTipText() {
    return "The password of the twitter user to use for connecting.";
  }

  /**
   * Returns the twitter connection object.
   *
   * @return		the connection
   */
  public twitter4j.Twitter getTwitterConnection() {
    twitter4j.Twitter	result;

    if (m_User.length() > 0)
      result = new TwitterFactory().getInstance(getUser(), getPassword().getValue());
    else
      result = new TwitterFactory().getInstance();

    return result;
  }

  /**
   * Returns the twitter stream connection object.
   *
   * @return		the stream connection
   */
  public twitter4j.TwitterStream getTwitterStreamConnection() {
    return new TwitterStreamFactory().getInstance(getUser(), getPassword().getValue());
  }

  /**
   * Executes the flow item.
   *
   * @return		always null
   */
  protected String doExecute() {
    return null;
  }
}
