/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * LowessPaintlet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.BasicStroke;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.util.ArrayList;
import java.util.Collections;

import adams.gui.core.GUIHelper;
import adams.gui.visualization.core.AntiAliasingPaintlet;
import adams.gui.visualization.stats.core.Point;

/**
 <!-- globalinfo-start -->
 * Paintlet for drawing the lowess overlay.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-stroke-thickness &lt;float&gt; (property: strokeThickness)
 * &nbsp;&nbsp;&nbsp;The thickness of the stroke.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * &nbsp;&nbsp;&nbsp;minimum: 0.01
 * </pre>
 *
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;Stroke color for the paintlet
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 *
 * <pre>-window-size &lt;int&gt; (property: windowSize)
 * &nbsp;&nbsp;&nbsp;The window size for smoothing.
 * &nbsp;&nbsp;&nbsp;default: 100
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author msf8
 * @version $Revision: 4584 $
 */
public class LowessPaintlet
  extends AbstractOverlayPaintlet
  implements AntiAliasingPaintlet {

  /** for serializing */
  private static final long serialVersionUID = 1643339689654875242L;

  /**Size of window size for calculating lowess */
  private int m_WindowSize;

  /**Points to plot for the lowess curve */
  protected ArrayList<Point> m_ToPlot;

  /** whether anti-aliasing is enabled. */
  protected boolean m_AntiAliasingEnabled;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Paintlet for drawing the lowess overlay.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"window-size", "windowSize",
	100, 1, null);

    m_OptionManager.add(
	    "anti-aliasing-enabled", "antiAliasingEnabled",
	    GUIHelper.getBoolean(getClass(), "antiAliasingEnabled", true));
  }

  public void setWindowSize(int val) {
    m_WindowSize = val;
    memberChanged();
  }

  /**
   * Get the Window size for calculating the lowess loverlay
   * @return		Number of data points in window
   */
  public int getWindowSize() {
    return m_WindowSize;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String windowSizeTipText() {
    return "The window size for smoothing.";
  }

  /**
   * Sets whether to use anti-aliasing.
   *
   * @param value	if true then anti-aliasing is used
   */
  public void setAntiAliasingEnabled(boolean value) {
    m_AntiAliasingEnabled = value;
    memberChanged();
  }

  /**
   * Returns whether anti-aliasing is used.
   *
   * @return		true if anti-aliasing is used
   */
  public boolean isAntiAliasingEnabled() {
    return m_AntiAliasingEnabled;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String antiAliasingEnabledTipText() {
    return "If enabled, uses anti-aliasing for drawing lines.";
  }

  public void calculate() {
    super.calculate();
    double[] x_data = m_Instances.attributeToDoubleArray(m_XInd);
    double[] y_data = m_Instances.attributeToDoubleArray(m_YInd);
    //create an arraylist of points from the instance data
    ArrayList<Point> points = new ArrayList<Point>();
    for(int i = 0; i< x_data.length; i++) {
      Point p = new Point(x_data[i], y_data[i]);
      points.add(p);
    }
    //sort the points on ascending x value
    Collections.sort(points);
    //This arraylist will contain the closest points to each data point.
    //size will be the window size
    ArrayList<Point> closest;
    //points to plot or the lowess curve
    m_ToPlot = new ArrayList<Point>();
    //If the number of data points is less than the window size specified
    if(m_WindowSize > points.size()) {
      m_WindowSize = points.size();
      System.out.println("Window size changed to number of points");
    }
    for(int i = 0; i< points.size(); i++) {
      closest = new ArrayList<Point>();
      closest.add(points.get(i));
      int index = 1;
      double ref = points.get(i).getX();
      int left = i -1;
      int right = i +1;
      while(index <m_WindowSize) {
	//if no points to the left
	if(left < 0) {
	  closest.add(points.get(right));
	  right ++;
	}
	//if no points to the right
	else if(right > points.size() -1) {
	  closest.add(points.get(left));
	  left --;
	}
	else {
	  //if point to the right is closer
	  if(Math.abs(points.get(right).getX()-ref) < Math.abs(points.get(left).getX() - ref)) {
	    closest.add(points.get(right));
	    right ++;
	  }
	  //if point to the left is closer
	  else {
	    closest.add(points.get(left));
	    left --;
	  }
	}
	index++;
      }

      //distance from the reference point of the furthest away point
      double max = Math.abs(closest.get(m_WindowSize -1).getX() - ref);

      double[] relDist = new double[closest.size()];
      //calculate the relative distances
      for(int j = 0; j<closest.size(); j++) {
	relDist[j] = Math.abs((closest.get(j).getX()-ref))/max;
      }

      //apply the tri-cube weight function
      double[] weighting = new double[relDist.length];
      for(int j = 0; j< relDist.length; j++) {
	weighting[j] = Math.pow((1-(Math.pow(relDist[j], 3.0))), 3.0);
      }

      //now fit a weighted least squares
      //based on code that excel uses for lowess

      double sumWts = 0;
      double sumWtX = 0;
      double sumWtX2 = 0;
      double sumWtY = 0;
      double sumWtXY = 0;

      for(int j = 0; j< weighting.length; j++) {
	sumWts += weighting[j];
	sumWtX += weighting[j] * closest.get(j).getX();
	sumWtX2 += weighting[j] * Math.pow(closest.get(j).getX(), 2.0);
	sumWtY += weighting[j] * closest.get(j).getY();
	sumWtXY += weighting[j] * closest.get(j).getY() * closest.get(j).getX();
      }
      double denom = sumWts * sumWtX2 - Math.pow(sumWtX, 2.0);

      double slope = (sumWts * sumWtXY - sumWtX * sumWtY)/denom;
      double intercept = (sumWtX2 * sumWtY - sumWtX * sumWtXY) /denom;

      double val = slope* closest.get(0).getX() + intercept;
      //add point calculated using weighted least squares
      m_ToPlot.add(new Point(closest.get(0).getX(), val));
    }
    //overlay calculated so now paint method can be run
    m_Calculated = true;
  }

  protected void drawData(Graphics g) {
    if(m_Calculated) {
      if (m_AntiAliasingEnabled)
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      else
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);
      g.setColor(m_Color);
      Graphics2D g2d = (Graphics2D)g;
      g2d.setStroke(new BasicStroke(m_StrokeThickness));
      //plot all the points
      for(int i = 0; i< m_ToPlot.size() -1; i++) {
	g2d.drawLine(m_AxisBottom.valueToPos(m_ToPlot.get(i).getX()), m_AxisLeft.valueToPos(m_ToPlot.get(i).getY()), m_AxisBottom.valueToPos(m_ToPlot.get(i+1).getX()), m_AxisLeft.valueToPos(m_ToPlot.get(i+1).getY()));
      }
    }
  }
}