/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Histogram.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.histogram;

import java.awt.BorderLayout;
import java.awt.Graphics;

import weka.core.Instances;
import adams.data.statistics.ArrayHistogram;
import adams.data.statistics.AbstractArrayStatistic.StatisticContainer;
import adams.data.statistics.ArrayHistogram.BinCalculation;
import adams.gui.visualization.core.PaintablePanel;
import adams.gui.visualization.core.PlotPanel;
import adams.gui.visualization.stats.paintlet.HistogramPaintlet;

/**
 * Class that displays a histogram displaying the data provided.
 *
 * @author msf8
 * @version $Revision: 4584 $
 */
public class Histogram
  extends PaintablePanel{

  /** for serialization */
  private static final long serialVersionUID = -4366437103496819542L;

  /**Instances to be plotted */
  protected Instances m_Instances;

  /** Panel for displaying the histogram */
  protected HistogramPanel m_Plot;

  /**Paintlet for plotting the data */
  protected HistogramPaintlet m_Val;

  /**Options for the histogram */
  protected HistogramOptions m_HistOptions;

  /** Position of the residuals attribute within the instances */
  protected int m_Index;

  /**
   * Set the options for this histogram
   * @param val			Histogramoptions object containing all the options
   */
  public void setOptions(HistogramOptions val) {
    m_HistOptions = val;
  }

  /**
   * Set the instances for the histogram
   * @param inst			Instances for the histogram plot
   */
  public void setInstances(Instances inst) {
    m_Instances = inst;
  }

  public PlotPanel getPlot() {
    return m_Plot;
  }

  public void prepareUpdate() {
  }

  protected boolean canPaint(Graphics g) {
    return(m_Plot != null);
  }

  /**
   * Called by the class that creates this histogram once
   * all the fields have been set
   */
  public void reset() {
    m_Val = m_HistOptions.m_Val;
    //Calculate bin size and frequency
    StatisticContainer cont;
    ArrayHistogram<Number> aHist = new ArrayHistogram<Number>();
    double[] data = m_Instances.attributeToDoubleArray(m_Index);
    Number[] numData = new Number[data.length];
    for(int i = 0; i< data.length; i++) {
      numData[i] = (Number)data[i];
    }
    aHist.add(numData);
    aHist.setBinWidth(m_HistOptions.m_WidthBin);
    aHist.setNumBins(m_HistOptions.m_NumBins);
    BinCalculation bc;
    //Set the bincalculation type
    if(m_HistOptions.m_BoxType.toString() == "DENSITY")
      bc = BinCalculation.DENSITY;
    else
      bc = BinCalculation.MANUAL;
    aHist.setBinCalculation(bc);
    //calculates depending on bin calculation type, will use
    //binwidth or numbins depending on type
    cont = aHist.calculate();
    int numBins = (Integer) cont.getMetaData(ArrayHistogram.METADATA_NUMBINS);
    //Start of intervals for bins
    double[] binX = (double[])cont.getMetaData(ArrayHistogram.METADATA_BINX);
    double binWidth = (Double) cont.getMetaData(ArrayHistogram.METADATA_BINWIDTH);
    double[][] plotdata = new double[numBins][2];
    //fill 2d array with positions of bins and count for each
    for (int i = 0; i < plotdata.length; i++) {
      plotdata[i][1] = (Double) cont.getCell(0, i);
      plotdata[i][0] = binX[i];
    }
    //Pass the data to the paintlet
    m_Val.setData(plotdata, binWidth, m_Instances.attribute(m_Index).name());
    m_Val.setPanel(this);
    repaint();
    revalidate();
  }

  protected void initGUI() {
    super.initGUI();
    setLayout(new BorderLayout());
    m_Val = new HistogramPaintlet();
    m_Plot = new HistogramPanel();
    m_Plot.addPaintListener(this);
    add(m_Plot, BorderLayout.CENTER);
    repaint();
  }

  /**
   * Set the index of residuals attribute within the instances
   * @param val			Position of residuals attribute
   */
  public void setIndex(int val) {
    m_Index = val;
  }
}