/*
 * AbstractZScorePaintlet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

import weka.core.Instances;
import adams.gui.core.GUIHelper;
import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AntiAliasingPaintlet;
import adams.gui.visualization.core.AxisPanel;
import adams.gui.visualization.core.plot.Axis;

/**
 * Abstract class for creating z score paintlets.
 *
 * @author msf8
 * @version $Revision: 4073 $
 */
public abstract class AbstractZScorePaintlet
  extends AbstractColorPaintlet
  implements AntiAliasingPaintlet {

  /** for serialization */
  private static final long serialVersionUID = 6918445466346742103L;

  /**index of the attribute being plotted */
  protected int m_Index;

  /**data from the attribute */
  protected double[] m_Data;

  /** y axis of plot */
  protected AxisPanel m_AxisLeft;

  /** x axis of plot */
  protected AxisPanel m_AxisBottom;

  /** the line color. */
  protected Color m_LineColor;

  /** whether anti-aliasing is enabled. */
  protected boolean m_AntiAliasingEnabled;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"line-color", "lineColor",
	Color.LIGHT_GRAY);

    m_OptionManager.add(
	"anti-aliasing-enabled", "antiAliasingEnabled",
	GUIHelper.getBoolean(getClass(), "antiAliasingEnabled", true));
  }

  protected void initialize() {
    super.initialize();
    m_Index = 0;
  }

  /**
   * Set the color to draw the lines.
   *
   * @param val		color for lines
   */
  public void setLineColor(Color val) {
    m_LineColor = val;
  }

  /**
   * Get the color for drawing the lines
   *
   * @return		color for lines
   */
  public Color getLineColor() {
    return m_LineColor;
  }

  /**
   * Tip text for the line color property.
   *
   * @return		String to describe the property
   */
  public String lineColorTipText() {
    return "Color for drawing the lines.";
  }

  /**
   * Sets whether to use anti-aliasing.
   *
   * @param value	if true then anti-aliasing is used
   */
  public void setAntiAliasingEnabled(boolean value) {
    m_AntiAliasingEnabled = value;
    memberChanged();
  }

  /**
   * Returns whether anti-aliasing is used.
   *
   * @return		true if anti-aliasing is used
   */
  public boolean isAntiAliasingEnabled() {
    return m_AntiAliasingEnabled;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String antiAliasingEnabledTipText() {
    return "If enabled, uses anti-aliasing for drawing lines.";
  }

  public PaintMoment getPaintMoment() {
    return PaintMoment.PAINT;
  }

  public void performPaint(Graphics g) {
    if(m_Instances != null)
      drawData(g);
  }

  /**set the index of the attribute */
  public void setIndex(int ind) {
    m_Index = ind;
    memberChanged();
  }

  /**
   * pass the required parameters for the paintlet
   * @param inst			Instances to be plotted
   * @param ind			index of the attribute
   */
  public void parameters(Instances inst, int ind) {
    m_Instances = inst;
    m_Index = ind;
  }

  protected void drawData(Graphics g) {
    if (m_AntiAliasingEnabled)
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
    else
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

    if (m_Instances != null) {
      g.setColor(Color.BLACK);

      m_Data = m_Instances.attributeToDoubleArray(m_Index);
      m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
      m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);
    }
  }
}