/*
 * Gamma.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.util.Arrays;

import org.apache.commons.math.MathException;
import org.apache.commons.math.distribution.GammaDistributionImpl;

import adams.gui.visualization.core.axis.Type;
import adams.gui.visualization.core.plot.Axis;

/**
 <!-- globalinfo-start -->
 * Paints the transformed gamma distribution
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-stroke-thickness &lt;float&gt; (property: strokeThickness)
 * &nbsp;&nbsp;&nbsp;The thickness of the stroke.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * &nbsp;&nbsp;&nbsp;minimum: 0.01
 * </pre>
 *
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;Stroke color for the paintlet
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 *
 * <pre>-size &lt;int&gt; (property: size)
 * &nbsp;&nbsp;&nbsp;Size of the data points
 * &nbsp;&nbsp;&nbsp;default: 5
 * </pre>
 *
 * <pre>-fill-point (property: fillPoint)
 * &nbsp;&nbsp;&nbsp;Whether to fill the data point with solid color
 * </pre>
 *
 * <pre>-fill-color &lt;java.awt.Color&gt; (property: fillColor)
 * &nbsp;&nbsp;&nbsp;Color for filling data point
 * &nbsp;&nbsp;&nbsp;default: #ff0000
 * </pre>
 *
 * <pre>-shape &lt;double&gt; (property: shape)
 * &nbsp;&nbsp;&nbsp;Shape paramter for gamma distribution
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 * <pre>-scale &lt;double&gt; (property: scale)
 * &nbsp;&nbsp;&nbsp;Scale parameter for gamma distribution
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author msf8
 * @version $Revision: 4073 $
 */
public class Gamma
extends AbstractProbabilityPaintlet{

  /** for serialization */
  private static final long serialVersionUID = 3439914415669101587L;

  /**Shape parameter for the gamma distribution */
  protected double m_Shape;

  /**Scale parameter for the gamma distribution */
  protected double m_Scale;

  protected void initialize() {
    super.initialize();
    m_Shape = -1.0;
    m_Scale = -1.0;
  }

  public void defineOptions() {
    super.defineOptions();

    //shape parameter
    m_OptionManager.add(
	"shape", "shape", 1.0);

    //scale parameter
    m_OptionManager.add(
	"scale", "scale", 1.0);
  }

  /**
   * Set the scale parameter for the gamma distribution
   * @param val			Scale parameter
   */
  public void setScale(double val) {
    m_Scale = val;
    memberChanged();
  }

  /**
   * Get the scale parameter for the gamma distribution
   * @return			Scale parameter
   */
  public double getScale() {
    return m_Scale;
  }

  /**
   * Tip text for the scale property
   * @return			String describing the property
   */
  public String scaleTipText() {
    return "Scale parameter for gamma distribution";
  }

  /**
   * Set the shape parameter for the gamma distribution
   * @param val			Shape parameter
   */
  public void setShape(double val) {
    m_Shape = val;
    memberChanged();
  }

  /**
   * Get the shape parameter for the gamma distribution
   * @return			Shape parameter
   */
  public double getShape() {
    return m_Shape;
  }

  /**
   * Tip text for the shape property
   * @return			String describing the property
   */
  public String shapeTipText() {
    return "Shape paramter for gamma distribution";
  }

  public void setAxis() {
    m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
    m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);

    m_AxisBottom.setType(Type.LOG_ABSOLUTE);
    m_AxisLeft.setType(Type.LOG_ABSOLUTE);
    m_AxisLeft.setNumberFormat("#.##");
  }

  public void performPaint(Graphics g) {
    if(m_Instances != null && m_Shape != -1.0) {
      if (m_AntiAliasingEnabled)
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      else
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

      double median;
      GammaDistributionImpl gam = new GammaDistributionImpl(m_Shape, m_Scale);
      double[] data = m_Instances.attributeToDoubleArray(m_Index);
      m_TransformedY = new double[data.length];
      Arrays.sort(data);
      for(int i = 0; i< data.length; i++) {
	median = ((i+1) -0.3)/(data.length + 0.4);
	try {
	  m_TransformedY[i] = gam.inverseCumulativeProbability(median);
	}
	catch (MathException e) {
	  e.printStackTrace();
	}
      }
      //If axis can handle the data
      if(m_AxisBottom.getType().canHandle(data[0], data[data.length-1])) {
	m_AxisBottom.setMinimum(data[0]);
	m_AxisBottom.setMaximum(data[data.length-1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }
      if(m_AxisLeft.getType().canHandle(m_TransformedY[0], m_TransformedY[m_TransformedY.length -1]))
      {
	m_AxisLeft.setMinimum(m_TransformedY[0]);
	m_AxisLeft.setMaximum(m_TransformedY[m_TransformedY.length -1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }
      m_AxisBottom.setAxisName(m_Instances.attribute(m_Index).name() + ")");
      m_AxisLeft.setAxisName("Inverse Gamma");

      for(int i = 0; i< data.length; i++) {
	Graphics2D g2d = (Graphics2D)g;
	//If data points are to be filled
	if(m_Fill) {
	  g2d.setColor(m_FillColor);
	  g2d.setStroke(new BasicStroke(0));
	  g2d.fillOval(m_AxisBottom.valueToPos(data[i])-m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i])-m_Size/2, m_Size, m_Size);
	}
	//outline of data point
	g2d.setStroke(new BasicStroke(m_StrokeThickness));
	g2d.setColor(m_Color);
	g2d.drawOval(m_AxisBottom.valueToPos(data[i])-m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i])-m_Size/2, m_Size, m_Size);
      }

      //If drawing regression fit diagonal
      if(m_RegressionLine) {
	g.setColor(Color.BLACK);
	double[] newData = new double[data.length];
	for(int i = 0; i < data.length; i++) {
	  newData[i] = Math.log(data[i]);
	}
	GammaDistributionImpl gd = new GammaDistributionImpl(m_Shape, m_Scale);
	//draw the expected diagonal line using the gamma distribution
	for(int i = 0; i< data.length-1; i++) {
	  double p1;
	  try {
	    p1 = gd.cumulativeProbability(newData[i]);
	  } catch (MathException e) {
	    p1 = 0;
	  }
	  double p2;
	  try {
	    p2 = gd.cumulativeProbability(newData[i+1]);
	  } catch (MathException e) {
	    p2 = 0;
	  }
	  g.drawLine(m_AxisBottom.valueToPos(data[i]), m_AxisLeft.valueToPos(p1), m_AxisBottom.valueToPos(data[i+1]), m_AxisLeft.valueToPos(p2));
	}
      }
    }
  }

  public String globalInfo() {
    return "Paints the transformed gamma distribution";
  }

  public boolean hasFitLine() {
    return false;
  }
}