/*
 * Normal.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.util.Arrays;

import JSci.maths.statistics.NormalDistribution;
import adams.data.statistics.StatUtils;
import adams.gui.visualization.core.axis.Type;
import adams.gui.visualization.core.plot.Axis;

/**
 <!-- globalinfo-start -->
 * Paints the transformed normal distribution on the panel
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-stroke-thickness &lt;float&gt; (property: strokeThickness)
 * &nbsp;&nbsp;&nbsp;The thickness of the stroke.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * &nbsp;&nbsp;&nbsp;minimum: 0.01
 * </pre>
 *
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;Stroke color for the paintlet
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 *
 * <pre>-size &lt;int&gt; (property: size)
 * &nbsp;&nbsp;&nbsp;Size of the data points
 * &nbsp;&nbsp;&nbsp;default: 5
 * </pre>
 *
 * <pre>-fill-point (property: fillPoint)
 * &nbsp;&nbsp;&nbsp;Whether to fill the data point with solid color
 * </pre>
 *
 * <pre>-fill-color &lt;java.awt.Color&gt; (property: fillColor)
 * &nbsp;&nbsp;&nbsp;Color for filling data point
 * &nbsp;&nbsp;&nbsp;default: #ff0000
 * </pre>
 *
 <!-- options-end -->
 *
 * @author msf8
 * @version $Revision: 4073 $
 */
public class Normal
extends AbstractProbabilityPaintlet{

  /** for serialization */
  private static final long serialVersionUID = -6765168739924987537L;

  public void setAxis() {
    m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
    m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);

    m_AxisBottom.setType(Type.ABSOLUTE);
    m_AxisLeft.setType(Type.ABSOLUTE);
    m_AxisLeft.setNumberFormat("#.##");
  }

  public void performPaint(Graphics g) {
    if(m_Instances != null) {
      if (m_AntiAliasingEnabled)
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      else
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

      //these two variables needed to use the normal distribution class
      //it works with real values when using the inverse normal rather
      //than z values
      int mean = 100;
      int var = 100;
      //estimated cumulative probability
      double median;
      NormalDistribution normal = new NormalDistribution(mean,var);
      double[] data = m_Instances.attributeToDoubleArray(m_Index);
      m_TransformedY = new double[data.length];
      Arrays.sort(data);
      for(int i = 0; i< data.length; i++) {
	median = ((i+1)-0.3)/(data.length+0.4);
	//Get the z value representing the probability
	//calculated using the chosen median formula
	m_TransformedY[i] = (normal.inverse(median)-mean)/Math.sqrt(var);
      }
      //if axis can handle the data
      if(m_AxisLeft.getType().canHandle(m_TransformedY[0], m_TransformedY[m_TransformedY.length-1]))
      {
	m_AxisLeft.setMinimum(m_TransformedY[0]);
	m_AxisLeft.setMaximum(m_TransformedY[m_TransformedY.length-1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }
      if(m_AxisBottom.getType().canHandle(data[0], data[data.length-1])) {
	m_AxisBottom.setMinimum(data[0]);
	m_AxisBottom.setMaximum(data[data.length-1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }

      m_AxisBottom.setAxisName(m_Instances.attribute(m_Index).name());
      m_AxisLeft.setAxisName("Inverse Normal");

      for(int i = 0; i< data.length; i++) {
	//plot the transformed value on the y axis against the original
	//data point on the x axis
	Graphics2D g2d = (Graphics2D)g;
	//If data points are to be filled
	if(m_Fill) {
	  g2d.setColor(m_FillColor);
	  g2d.setStroke(new BasicStroke(0));
	  g2d.fillOval(m_AxisBottom.valueToPos(data[i]) -m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i]) -m_Size/2, m_Size, m_Size);
	}
	//Outline of data point
	g2d.setStroke(new BasicStroke(m_StrokeThickness));
	g2d.setColor(m_Color);
	g2d.drawOval(m_AxisBottom.valueToPos(data[i]) -m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i]) -m_Size/2, m_Size, m_Size);

      }
      //if drawing regression fit diagonal
      if(m_RegressionLine) {
	g.setColor(Color.BLACK);
	double mn = StatUtils.mean(data);
	double std = StatUtils.stddev(data, false);
	for(int i = 0; i< data.length-1; i++) {
	  double p1 = (data[i]-mn)/std;
	  double p2 = (data[i+1]-mn)/std;
	  g.drawLine(m_AxisBottom.valueToPos(data[i]), m_AxisLeft.valueToPos(p1), m_AxisBottom.valueToPos(data[i+1]), m_AxisLeft.valueToPos(p2));
	}
      }
    }
  }

  public String globalInfo() {
    return " Paints the transformed normal distribution on the panel";
  }

  public boolean hasFitLine() {
    return true;
  }
}