/*
 * AbstractProbabilityPaintlet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.Color;

import adams.gui.core.GUIHelper;
import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AntiAliasingPaintlet;
import adams.gui.visualization.core.AxisPanel;

/**
 * Abstract class for paintlets that plot the regression in the
 * probability plot.
 *
 * @author msf8
 * @version $Revision: 4073 $
 */
public abstract class AbstractProbabilityPaintlet
  extends AbstractColorPaintlet
  implements AntiAliasingPaintlet {

  /** for serialization */
  private static final long serialVersionUID = 6017537002376582174L;

  /** Index of the attribute in the instances */
  protected int m_Index;

  /**Whether to draw a regression line */
  protected boolean m_RegressionLine;

  /** Size of the points */
  protected int m_Size;

  /**Whether to fill the data points */
  protected boolean m_Fill;

  /**x axis of the plot */
  protected AxisPanel m_AxisLeft;

  /**y axis of the plot */
  protected AxisPanel m_AxisBottom;

  /**Transformed data for the y axis */
  protected double[] m_TransformedY;

  /** Color to fill data points */
  protected Color m_FillColor;

  /** whether anti-aliasing is enabled. */
  protected boolean m_AntiAliasingEnabled;

  /**
   * Sets the axis to the correct type for the distribution
   */
  public abstract void setAxis();


  public PaintMoment getPaintMoment() {
    return PaintMoment.PAINT;
  }

  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"size", "size", 5);

    m_OptionManager.add(
	"fill-point", "fillPoint", true);

    m_OptionManager.add(
	"fill-color", "fillColor", Color.RED);

    m_OptionManager.add(
	"anti-aliasing-enabled", "antiAliasingEnabled",
	GUIHelper.getBoolean(getClass(), "antiAliasingEnabled", true));
  }

  /**
   * Set the color for filling the data points
   * @param val		Color for filling points
   */
  public void setFillColor(Color val) {
    m_FillColor = val;
    memberChanged();
  }

  /**
   * get the color for filling the data points
   * @return			Color for filling the data points
   */
  public Color getFillColor() {
    return m_FillColor;
  }

  /**
   * Tip text for the fill color property
   * @return			String describing the property
   */
  public String fillColorTipText() {
    return "Color for filling data point";
  }

  /**
   * Set whether the data points are filled with color
   * @param val			True if data points filled
   */
  public void setFillPoint(boolean val) {
    m_Fill = val;
    memberChanged();
  }

  /**
   * Get whether the data points should be filled with color
   * @return			True if filled
   */
  public boolean getFillPoint() {
    return m_Fill;
  }

  /**
   * Tip text for the fill point property
   * @return			String describing the property
   */
  public String fillPointTipText() {
    return "Whether to fill the data point with solid color";
  }

  /**
   * Set the size of the data points
   * @param val		Size of the data points in pixels
   */
  public void setSize(int val){
    m_Size = val;
    memberChanged();
  }

  /**
   * Get the size of the data points
   * @return			Size of the points in pixels
   */
  public int getSize() {
    return m_Size;
  }

  /**
   * Tip text for the data point size property
   * @return
   */
  public String sizeTipText() {
    return "Size of the data points";
  }

  /**
   * Sets whether to use anti-aliasing.
   *
   * @param value	if true then anti-aliasing is used
   */
  public void setAntiAliasingEnabled(boolean value) {
    m_AntiAliasingEnabled = value;
    memberChanged();
  }

  /**
   * Returns whether anti-aliasing is used.
   *
   * @return		true if anti-aliasing is used
   */
  public boolean isAntiAliasingEnabled() {
    return m_AntiAliasingEnabled;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String antiAliasingEnabledTipText() {
    return "If enabled, uses anti-aliasing for drawing lines.";
  }

  /**
   * Set the index of the attribute used
   * @param val			Attribute index
   */
  public void setIndex(int val) {
    m_Index = val;
    memberChanged();
  }

  /**
   * Set whether a regression line is drawn
   * @param val			True if regression line is to be drawn
   */
  public void setLine(boolean val) {
    m_RegressionLine = val;
    memberChanged();
  }

  /**
   * Whether a regression line has been implemented
   * for this regresion paintlet
   * @return			True if regression line implemented
   */
  public abstract boolean hasFitLine();
}