/*
 * AbstractScatterPlotPaintlet.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

import adams.core.option.OptionUtils;
import adams.gui.core.GUIHelper;
import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AntiAliasingPaintlet;
import adams.gui.visualization.core.AxisPanel;
import adams.gui.visualization.core.plot.Axis;

/**
 * Abstract class for creating scatterplot paintlets.
 *
 * @author msf8
 * @version $Revision: 4073 $
 */
public abstract class AbstractScatterPlotPaintlet
  extends AbstractColorPaintlet
  implements AntiAliasingPaintlet {

  /** for serialization */
  private static final long serialVersionUID = 7191423312364530577L;

  /**index of attribute for x axis */
  protected int m_XIndex;

  /**Index of attribute for y axis */
  protected int m_YIndex;

  /** size of the plot points */
  protected int m_Size;

  /**Data to display on the x axis */
  protected double[] m_XData;

  /**Data to display on the y axis */
  protected double[] m_YData;

  /** y axis of plot */
  protected AxisPanel m_AxisLeft;

  /** x axis of plot */
  protected AxisPanel m_AxisBottom;

  /** whether anti-aliasing is enabled. */
  protected boolean m_AntiAliasingEnabled;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "anti-aliasing-enabled", "antiAliasingEnabled",
	    GUIHelper.getBoolean(getClass(), "antiAliasingEnabled", true));
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_XIndex = 0;
    m_YIndex = 0;
  }

  /**
   * Sets whether to use anti-aliasing.
   *
   * @param value	if true then anti-aliasing is used
   */
  public void setAntiAliasingEnabled(boolean value) {
    m_AntiAliasingEnabled = value;
    memberChanged();
  }

  /**
   * Returns whether anti-aliasing is used.
   *
   * @return		true if anti-aliasing is used
   */
  public boolean isAntiAliasingEnabled() {
    return m_AntiAliasingEnabled;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String antiAliasingEnabledTipText() {
    return "If enabled, uses anti-aliasing for drawing lines.";
  }

  public PaintMoment getPaintMoment() {
    return PaintMoment.PAINT;
  }

  public void performPaint(Graphics g) {
    if(m_Instances != null)
      drawData(g);
  }

  /**
   * Returns info on scatterpaintlet object
   */
  public String toString() {
    return OptionUtils.getCommandLine(this);
  }

  /**
   * draws the data on the graphics object
   * @param g		Graphics object to draw on
   */
  protected void drawData(Graphics g) {
    if(m_Instances != null) {
      g.setColor(m_Color);

      if (m_AntiAliasingEnabled)
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      else
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

      //arrays of data to be plotted
      m_XData = m_Instances.attributeToDoubleArray(m_XIndex);
      m_YData = m_Instances.attributeToDoubleArray(m_YIndex);

      m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
      m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);
    }
  }

  /**
   * get index of chosen attribute for x axis
   * @return		chosen index
   */
  public int getX_Index()
  {
    return m_XIndex;
  }
  /**
   * Set the index of attribute for x axis
   * @param val		Index to set
   */
  public void setX_Index(int val)
  {
    m_XIndex = val;
    memberChanged();
  }
  /**
   * Get index of chosen attribute for y axis
   * @return		chosen index
   */
  public int getY_Index()
  {
    return m_YIndex;
  }
  /**
   * Set the index of attribute for y axis
   * @param val		Index to set
   */
  public void setY_Index(int val)
  {
    m_YIndex = val;
    memberChanged();
  }
}